------------------------------------------------------------------------------
--
--	Licensed Materials - Property of IBM
--
--	5724O4800
--
--	(C) Copyright IBM Corp. 2017. All Rights Reserved
--
--	US Government Users Restricted Rights - Use, duplication
--	or disclosure restricted by GSA ADP Schedule Contract
--	with IBM Corp.
--
--	Netcool/OMNIbus 8.1.0 - July 2018  Build: 5.50.78
--
--	SCOPE BASED EVENT GROUPING
--
--	This SQL file updates code used by scope-based event grouping
--
------------------------------------------------------------------------------

------------------------------------------------------------------------------
-- UPDATE THE VERSION OF SCOPE BASED EVENT GROUPING
------------------------------------------------------------------------------

UPDATE master.properties set CharValue = 'Netcool/OMNIbus 8.1.0 - July 2018  Build: 5.50.78' WHERE Name = 'SEGVersion';
go

------------------------------------------------------------------------------
-- EXTEND THE SIZE OF THE CUSTOM TEXT FIELD
------------------------------------------------------------------------------

ALTER TABLE alerts.status ALTER COLUMN CustomText SET WIDTH 4096;
go

ALTER TABLE master.correlation_sitename ALTER COLUMN CustomText SET WIDTH 4096;
go

ALTER TABLE master.correlation_scopeid ALTER COLUMN CustomText SET WIDTH 4096;
go

------------------------------------------------------------------------------
-- CREATE TABLE TO STORE PRIORITY CHILD EVENT WEIGHT DATA
------------------------------------------------------------------------------

CREATE TABLE master.correlation_priority_children PERSISTENT
(
	Identifier	VARCHAR(255) PRIMARY KEY,
	CustomText	VARCHAR(4096),
	CustomTimestamp	TIME,
	CustomWeight	INTEGER,
	HighImpactWeight	INTEGER,
	HighImpactText	VARCHAR(255),
	HighCauseWeight	INTEGER,
	HighCauseText	VARCHAR(255)
);
go

------------------------------------------------------------------------------
-- CREATE TABLE TO STORE POLICIES CREATED BY THE UI
------------------------------------------------------------------------------

CREATE TABLE master.correlation_scope_based_policies PERSISTENT (
	PolicyId VARCHAR(255) PRIMARY KEY,
	PolicyData VARCHAR(8192),
	CreatedBy VARCHAR(255),
	TriggerName VARCHAR(255),
	LastRun TIME
);
go 

------------------------------------------------------------------------------
-- CREATE NEW PROPERTIES TO BE USED BY THE AUTOMATIONS
------------------------------------------------------------------------------

-- SPECIFIES WHETHER OR NOT TO JOURNAL CHILD EVENTS TO THE SUPER PARENT EVENT
insert into master.properties (Name, IntValue) values ('SEGJournalToSuperParent', 0);
go

-- SPECIFIES MAXIMUM NUMBER OF EVENTS TO SEND TO THE SUPER PARENT EVENT JOURNAL
insert into master.properties (Name, IntValue) values ('SEGMaxSuperParentJournals', 100);
go

------------------------------------------------------------------------------
-- CREATE INSERT TRIGGER ON alerts.status TO CREATE PARENT EVENTS AND LINK CHILDREN
------------------------------------------------------------------------------

CREATE OR REPLACE TRIGGER correlation_new_row
GROUP correlation_triggers
PRIORITY 15
COMMENT 'Checks for the existence of parent events, updates or creates if necessary'
BEFORE INSERT ON alerts.status
FOR EACH ROW
WHEN get_prop_value('ActingPrimary') %= 'TRUE' and
	new.ScopeID != '' and
	new.ParentIdentifier = '' and
	new.AlertGroup not in ('SiteNameParent', 'ScopeIDParent', 'Synthetic Event - Parent', 'ASMParent')
declare

	quietperiod		INTEGER;
	usenodeforscopeidparent	INTEGER;
	usenodeforsitenameparent INTEGER;
	nodetouse		CHAR(64);
	nositenameparentifsitenameblank INTEGER;

	now			TIME;
	expiretime		TIME;

	scopeid			CHAR(255);

	scopeparentfound	INTEGER;
	newscopeparentneeded	INTEGER;
	scopeidentifier		CHAR(255);

	siteparentfound		INTEGER;
	scopeparentmoved	INTEGER;
	siteidentifier		CHAR(255);

begin

-- STEP 1: INITIALISE VARIABLES

	set quietperiod = 15 * 60;
	set usenodeforscopeidparent = 0;
	set usenodeforsitenameparent = 0;
	set nodetouse = '';
	set nositenameparentifsitenameblank = 0;

	set now = getdate();

	set scopeid = '';

	set scopeparentfound = 0;
	set newscopeparentneeded = 0;
	set scopeidentifier = '';

	set siteparentfound = 0;
	set scopeparentmoved = 0;
	set siteidentifier = '';

	-- SET UP VARIABLES BASED ON PROPERTIES
	for each row property in master.properties where property.Name in (
		'SEGQuietPeriod',
		'SEGUseNodeForScopeIDParent',
		'SEGUseNodeForSiteNameParent',
		'SEGNoSiteNameParentIfSiteNameBlank')
	begin
		-- SPECIFIES THE GLOBAL DEFAULT QUIET PERIOD TO APPLY TO GROUPING
		if (property.Name = 'SEGQuietPeriod') then

			set quietperiod = property.IntValue;

		-- SPECIFIES IF THE NODE VALUE OF THE FIRST EVENT SHOULD BE USED FOR THE ScopeIDParent EVENT
		elseif (property.Name = 'SEGUseNodeForScopeIDParent') then

			set usenodeforscopeidparent = property.IntValue;

		-- SPECIFIES IF THE NODE VALUE OF THE FIRST EVENT SHOULD BE USED FOR THE SiteNameParent EVENT
		elseif (property.Name = 'SEGUseNodeForSiteNameParent') then

			set usenodeforsitenameparent = property.IntValue;

		-- SPECIFIES WHETHER OR NOT A SiteNameParent SHOULD BE CREATED IF SiteName IS NULL
		elseif (property.Name = 'SEGNoSiteNameParentIfSiteNameBlank') then

			set nositenameparentifsitenameblank = property.IntValue;

		end if;
	end;

	-- OVERRIDE QUIET PERIOD IF DEFINED IN THE INCOMING EVENT
	if (new.QuietPeriod != 0) then

		set quietperiod = new.QuietPeriod;
	end if;

	-- USE VALUE STORED IN FirstOccurrence IF SET, ELSE USE CURRENT TIME
	if (new.FirstOccurrence = 0) then

		set now = getdate();
	else 

		set now = new.FirstOccurrence;
	end if;

	-- SET DEFAULT VALUE OF ExpireTime BASED ON FirstOccurrence
	set expiretime = now + quietperiod;

-- STEP 2: CHECK ScopeAlias MEMBERSHIP

	-- CHECK FOR ScopeAlias MEMBERSHIP
	for each row scopealiasmemberrow in master.correlation_scopealias_members where
		scopealiasmemberrow.ScopeID = new.ScopeID
	begin

		-- SET THE ScopeID TO USE FOR THE SYNTHETIC CONTAINMENT EVENT TO BE THE ALIAS
		set scopeid = scopealiasmemberrow.ScopeAlias;
	end;

	-- USE INCOMING ScopeID VALUE IF ScopeID IS NOT A MEMBER OF AN ALIAS GROUP
	if (scopeid = '') then

		set scopeid = new.ScopeID;
	end if;

-- STEP 3: SET UP ScopeIDParent EVENT

	-- LOOK FOR CURRENT ScopeIDParent
	for each row parent in master.correlation_scopeid where
		parent.ScopeID = scopeid
	begin

		-- MARK ScopeID PARENT AS FOUND
		set scopeparentfound = 1;

		-- CHECK IF THE GROUP ExpireTime HAS PASSED
		if (parent.ExpireTime < now) then

			-- IF SO, UPDATE PARENT ENTRY AND SET FLAG TO CREATE A NEW ScopeIDParent
			set parent.Identifier =	'SIDP:' + to_char(to_int(now)) + ':' + parent.ScopeID;
			set newscopeparentneeded = 1;
			set parent.ExpireTime = expiretime;
			set parent.CustomText =	'';
			set parent.CustomTimestamp = 0;
			set parent.CustomWeight = 0;
			set parent.HighImpactWeight = 0;
			set parent.HighImpactText = '';
			set parent.HighCauseWeight = 0;
			set parent.HighCauseText = '';
		end if;

		-- STORE EITHER EXISTING OR UPDATED ScopeIDParent IDENTIFIER
		set scopeidentifier = parent.Identifier;

		-- UPDATE LastOccurrence IN PARENT ENTRY IF INCOMING VALUE IS GREATER
		if (parent.LastOccurrence < now) then

			set parent.LastOccurrence = now;
		end if;

		-- UPDATE GROUP ExpireTime IF INCOMING ROW EXTENDS IT
		-- DO NOT EXTEND ExpireTime IF SCOPEID IS MARKED AS HAVING A FIXED TIME WINDOW
		-- FIXED TIME WINDOW SCOPEIDS ARE PREFIXED WITH THE STRING FX:
		if (parent.ExpireTime < expiretime and parent.ScopeID not like '^FX:') then

			set parent.ExpireTime = expiretime;
		end if;
	end;

	-- CREATE PARENT ENTRY IN master.correlation_scopeid IF NONE FOUND
	if (scopeparentfound = 0) then

		-- SET ScopeIDParent IDENTIFIER TO A NEW VALUE
		set scopeidentifier = 'SIDP:' + to_char(to_int(now)) + ':' + scopeid;

		-- INSERT NEW ScopeIDParent
		insert into master.correlation_scopeid (
			ScopeID,
			LastOccurrence,
			Identifier,
			ExpireTime)
		values (
			scopeid,
			now,
			scopeidentifier,
			expiretime);
	end if;

	-- SET THE NODE FIELD VALUE TO USE BASED ON PROPERTY
	if (usenodeforscopeidparent = 1) then

		set nodetouse = new.Node;
	else

		set nodetouse = scopeid;
	end if;

	-- INSERT ScopeIDParent EVENT INTO alerts.status IF NOT PRESENT OR NEW ONE NEEDED
	if (scopeparentfound = 0 or newscopeparentneeded = 1) then

		-- INSERT SYNTHETIC ScopeIDParent EVENT
		insert into alerts.status (
			Identifier,
			Node,
			Class,
			Summary,
			AlertGroup,
			Severity,
			ScopeID,
			FirstOccurrence,
			LastOccurrence,
			Grade,
			OwnerUID,
			OwnerGID)
		values (
			scopeidentifier,
			nodetouse,
			99990,
			'INCIDENT: ' + scopeid + ': calculating summary details...',
			'ScopeIDParent',
			1,
			scopeid,
			now,
			now,
			1,
			65534,
			0);
	end if;

-- STEP 4: SET UP SiteNameParent EVENT IF OPTION TO DISABLE IS NOT SET

	-- DO NOT CREATE SiteNameParent EVENT IF OPTION TO DISABLE SiteNameParent
	-- IS SET AND SiteName IS BLANK.  INSTEAD LINK EVENT TO ScopeIDParent EVENT
	if (nositenameparentifsitenameblank = 1 and new.SiteName = '') then

		-- LINK CURRENT EVENT TO ScopeIDParent EVENT
		set new.ParentIdentifier = scopeidentifier;

	-- ELSE CREATE SiteNameParent EVENT AND LINK CURRENT EVENT TO THAT
	else

		-- IF EVENT HAS NO SiteName THEN SET IT TO A DEFAULT
		if (new.SiteName = '') then

			set new.SiteName = 'NO SITENAME';
		end if;

		-- LOOK FOR CURRENT SiteNameParent
		for each row parent in master.correlation_sitename where
			parent.SiteName = new.SiteName and
			parent.ScopeID = scopeid
		begin

			-- MARK SiteName PARENT AS FOUND
			set siteparentfound = 1;

			-- CHECK IF ScopeIDParent HAS MOVED
			for each row scopeidparentevent in alerts.status where
				scopeidparentevent.Identifier = parent.Identifier
			begin

				if (scopeidparentevent.ParentIdentifier != scopeidentifier) then

					set scopeparentmoved = 1;
				end if;
			end;

			-- UPDATE THE ENTRY IF WE ARE CREATING A NEW ScopeIDParent
			if (newscopeparentneeded = 1 or scopeparentmoved = 1) then

				set parent.Identifier = 'SNP:' + to_char(to_int(now)) + ':' + scopeid + ':' + new.SiteName;
				set parent.CustomText =	'';
				set parent.CustomTimestamp = 0;
				set parent.CustomWeight = 0;
				set parent.HighImpactWeight = 0;
				set parent.HighImpactText = '';
				set parent.HighCauseWeight = 0;
				set parent.HighCauseText = '';
			end if;

			-- STORE CURRENT SiteName PARENT IDENTIFIER
			set siteidentifier = parent.Identifier;
		end;

		-- CREATE PARENT ENTRY IN master.correlation_sitename IF NONE FOUND
		if (siteparentfound = 0) then

			-- STORE NEW SiteName PARENT IDENTIFIER
			set siteidentifier = 'SNP:' +
				to_char(to_int(now)) + ':' + scopeid + ':' + new.SiteName;

			-- CREATE THE NEW SiteParent EVENT
			insert into master.correlation_sitename (
				SiteName,
				ScopeID,
				Identifier)
			values (
				new.SiteName,
				scopeid,
				siteidentifier);
		end if;

		-- SET THE NODE FIELD VALUE TO USE BASED ON PROPERTY
		if (usenodeforsitenameparent = 1) then

			set nodetouse = new.Node;
		else

			set nodetouse = new.SiteName;
		end if;

		-- INSERT NEW SiteParent EVENT INTO alerts.status IF NEW OR UPDATED ScopeIDParent
		if (siteparentfound = 0 or newscopeparentneeded = 1 or scopeparentmoved = 1) then

			-- INSERT SYNTHETIC SiteNameParent EVENT
			insert into alerts.status (
				Identifier,
				Node,
				Class,
				Summary,
				AlertGroup,
				Severity,
				ScopeID,
				SiteName,
				FirstOccurrence,
				LastOccurrence,
				ParentIdentifier,
				Grade,
				OwnerUID,
				OwnerGID)
			values (
				siteidentifier,
				nodetouse,
				99990,
				new.SiteName + ': calculating summary details...',
				'SiteNameParent',
				1,
				scopeid,
				new.SiteName,
				now,
				now,
				scopeidentifier,
				1,
				65534,
				0);
		end if;

		-- LINK CURRENT EVENT TO SiteNameParent
		set new.ParentIdentifier = siteidentifier;
	end if;
end;
go

------------------------------------------------------------------------------
-- CREATE UPDATE TRIGGER ON alerts.status TO CREATE PARENT EVENTS AND LINK CHILDREN
------------------------------------------------------------------------------

CREATE OR REPLACE TRIGGER correlation_update
GROUP correlation_triggers
PRIORITY 15
COMMENT 'Checks for the existence of parent events, updates or creates if necessary'
BEFORE UPDATE ON alerts.status
FOR EACH ROW
WHEN get_prop_value('ActingPrimary') %= 'TRUE' and
	old.ScopeID = '' and new.ScopeID != '' and
	old.ParentIdentifier = '' and
	new.AlertGroup not in ('SiteNameParent', 'ScopeIDParent', 'Synthetic Event - Parent', 'ASMParent')
declare

	quietperiod		INTEGER;
	usenodeforscopeidparent	INTEGER;
	usenodeforsitenameparent INTEGER;
	nodetouse		CHAR(64);
	nositenameparentifsitenameblank INTEGER;

	now			TIME;
	expiretime		TIME;

	scopeid			CHAR(255);

	scopeparentfound	INTEGER;
	newscopeparentneeded	INTEGER;
	scopeidentifier		CHAR(255);

	siteparentfound		INTEGER;
	scopeparentmoved	INTEGER;
	siteidentifier		CHAR(255);

begin

-- STEP 1: INITIALISE VARIABLES

	set quietperiod = 15 * 60;
	set usenodeforscopeidparent = 0;
	set usenodeforsitenameparent = 0;
	set nodetouse = '';
	set nositenameparentifsitenameblank = 0;

	set now = getdate();

	set scopeid = '';

	set scopeparentfound = 0;
	set newscopeparentneeded = 0;
	set scopeidentifier = '';

	set siteparentfound = 0;
	set scopeparentmoved = 0;
	set siteidentifier = '';

	-- SET UP VARIABLES BASED ON PROPERTIES
	for each row property in master.properties where property.Name in (
		'SEGQuietPeriod',
		'SEGUseNodeForScopeIDParent',
		'SEGUseNodeForSiteNameParent',
		'SEGNoSiteNameParentIfSiteNameBlank')
	begin
		-- SPECIFIES THE GLOBAL DEFAULT QUIET PERIOD TO APPLY TO GROUPING
		if (property.Name = 'SEGQuietPeriod') then

			set quietperiod = property.IntValue;

		-- SPECIFIES IF THE NODE VALUE OF THE FIRST EVENT SHOULD BE USED FOR THE ScopeIDParent EVENT
		elseif (property.Name = 'SEGUseNodeForScopeIDParent') then

			set usenodeforscopeidparent = property.IntValue;

		-- SPECIFIES IF THE NODE VALUE OF THE FIRST EVENT SHOULD BE USED FOR THE SiteNameParent EVENT
		elseif (property.Name = 'SEGUseNodeForSiteNameParent') then

			set usenodeforsitenameparent = property.IntValue;

		-- SPECIFIES WHETHER OR NOT A SiteNameParent SHOULD BE CREATED IF SiteName IS NULL
		elseif (property.Name = 'SEGNoSiteNameParentIfSiteNameBlank') then

			set nositenameparentifsitenameblank = property.IntValue;

		end if;
	end;

	-- OVERRIDE QUIET PERIOD IF DEFINED IN THE INCOMING EVENT
	if (new.QuietPeriod != 0) then

		set quietperiod = new.QuietPeriod;
	end if;

	-- USE VALUE STORED IN FirstOccurrence IF SET, ELSE USE CURRENT TIME
	if (new.FirstOccurrence = 0) then

		set now = getdate();
	else 

		set now = new.FirstOccurrence;
	end if;

	-- SET DEFAULT VALUE OF ExpireTime BASED ON FirstOccurrence
	set expiretime = now + quietperiod;

-- STEP 2: CHECK ScopeAlias MEMBERSHIP

	-- CHECK FOR ScopeAlias MEMBERSHIP
	for each row scopealiasmemberrow in master.correlation_scopealias_members where
		scopealiasmemberrow.ScopeID = new.ScopeID
	begin

		-- SET THE ScopeID TO USE FOR THE SYNTHETIC CONTAINMENT EVENT TO BE THE ALIAS
		set scopeid = scopealiasmemberrow.ScopeAlias;
	end;

	-- USE INCOMING ScopeID VALUE IF ScopeID IS NOT A MEMBER OF AN ALIAS GROUP
	if (scopeid = '') then

		set scopeid = new.ScopeID;
	end if;

-- STEP 3: SET UP ScopeIDParent EVENT

	-- LOOK FOR CURRENT ScopeIDParent
	for each row parent in master.correlation_scopeid where
		parent.ScopeID = scopeid
	begin

		-- MARK ScopeID PARENT AS FOUND
		set scopeparentfound = 1;

		-- CHECK IF THE GROUP ExpireTime HAS PASSED
		if (parent.ExpireTime < now) then

			-- IF SO, UPDATE PARENT ENTRY AND SET FLAG TO CREATE A NEW ScopeIDParent
			set parent.Identifier =	'SIDP:' + to_char(to_int(now)) + ':' + parent.ScopeID;
			set newscopeparentneeded = 1;
			set parent.ExpireTime = expiretime;
			set parent.CustomText =	'';
			set parent.CustomTimestamp = 0;
			set parent.CustomWeight = 0;
			set parent.HighImpactWeight = 0;
			set parent.HighImpactText = '';
			set parent.HighCauseWeight = 0;
			set parent.HighCauseText = '';
		end if;

		-- STORE EITHER EXISTING OR UPDATED ScopeIDParent IDENTIFIER
		set scopeidentifier = parent.Identifier;

		-- UPDATE LastOccurrence IN PARENT ENTRY IF INCOMING VALUE IS GREATER
		if (parent.LastOccurrence < now) then

			set parent.LastOccurrence = now;
		end if;

		-- UPDATE GROUP ExpireTime IF INCOMING ROW EXTENDS IT
		-- DO NOT EXTEND ExpireTime IF SCOPEID IS MARKED AS HAVING A FIXED TIME WINDOW
		-- FIXED TIME WINDOW SCOPEIDS ARE PREFIXED WITH THE STRING FX:
		if (parent.ExpireTime < expiretime and parent.ScopeID not like '^FX:') then

			set parent.ExpireTime = expiretime;
		end if;
	end;

	-- CREATE PARENT ENTRY IN master.correlation_scopeid IF NONE FOUND
	if (scopeparentfound = 0) then

		-- SET ScopeIDParent IDENTIFIER TO A NEW VALUE
		set scopeidentifier = 'SIDP:' + to_char(to_int(now)) + ':' + scopeid;

		-- INSERT NEW ScopeIDParent
		insert into master.correlation_scopeid (
			ScopeID,
			LastOccurrence,
			Identifier,
			ExpireTime)
		values (
			scopeid,
			now,
			scopeidentifier,
			expiretime);
	end if;

	-- SET THE NODE FIELD VALUE TO USE BASED ON PROPERTY
	if (usenodeforscopeidparent = 1) then

		set nodetouse = new.Node;
	else

		set nodetouse = scopeid;
	end if;

	-- INSERT ScopeIDParent EVENT INTO alerts.status IF NOT PRESENT OR NEW ONE NEEDED
	if (scopeparentfound = 0 or newscopeparentneeded = 1) then

		-- INSERT SYNTHETIC ScopeIDParent EVENT
		insert into alerts.status (
			Identifier,
			Node,
			Class,
			Summary,
			AlertGroup,
			Severity,
			ScopeID,
			FirstOccurrence,
			LastOccurrence,
			Grade,
			OwnerUID,
			OwnerGID)
		values (
			scopeidentifier,
			nodetouse,
			99990,
			'INCIDENT: ' + scopeid + ': calculating summary details...',
			'ScopeIDParent',
			1,
			scopeid,
			now,
			now,
			1,
			65534,
			0);
	end if;

-- STEP 4: SET UP SiteNameParent EVENT IF OPTION TO DISABLE IS NOT SET

	-- DO NOT CREATE SiteNameParent EVENT IF OPTION TO DISABLE SiteNameParent
	-- IS SET AND SiteName IS BLANK.  INSTEAD LINK EVENT TO ScopeIDParent EVENT
	if (nositenameparentifsitenameblank = 1 and new.SiteName = '') then

		-- LINK CURRENT EVENT TO ScopeIDParent EVENT
		set new.ParentIdentifier = scopeidentifier;

	-- ELSE CREATE SiteNameParent EVENT AND LINK CURRENT EVENT TO THAT
	else

		-- IF EVENT HAS NO SiteName THEN SET IT TO A DEFAULT
		if (new.SiteName = '') then

			set new.SiteName = 'NO SITENAME';
		end if;

		-- LOOK FOR CURRENT SiteNameParent
		for each row parent in master.correlation_sitename where
			parent.SiteName = new.SiteName and
			parent.ScopeID = scopeid
		begin

			-- MARK SiteName PARENT AS FOUND
			set siteparentfound = 1;

			-- CHECK IF ScopeIDParent HAS MOVED
			for each row scopeidparentevent in alerts.status where
				scopeidparentevent.Identifier = parent.Identifier
			begin

				if (scopeidparentevent.ParentIdentifier != scopeidentifier) then

					set scopeparentmoved = 1;
				end if;
			end;

			-- UPDATE THE ENTRY IF WE ARE CREATING A NEW ScopeIDParent
			if (newscopeparentneeded = 1 or scopeparentmoved = 1) then

				set parent.Identifier = 'SNP:' + to_char(to_int(now)) + ':' + scopeid + ':' + new.SiteName;
				set parent.CustomText =	'';
				set parent.CustomTimestamp = 0;
				set parent.CustomWeight = 0;
				set parent.HighImpactWeight = 0;
				set parent.HighImpactText = '';
				set parent.HighCauseWeight = 0;
				set parent.HighCauseText = '';
			end if;

			-- STORE CURRENT SiteName PARENT IDENTIFIER
			set siteidentifier = parent.Identifier;
		end;

		-- CREATE PARENT ENTRY IN master.correlation_sitename IF NONE FOUND
		if (siteparentfound = 0) then

			-- STORE NEW SiteName PARENT IDENTIFIER
			set siteidentifier = 'SNP:' +
				to_char(to_int(now)) + ':' + scopeid + ':' + new.SiteName;

			-- CREATE THE NEW SiteParent EVENT
			insert into master.correlation_sitename (
				SiteName,
				ScopeID,
				Identifier)
			values (
				new.SiteName,
				scopeid,
				siteidentifier);
		end if;

		-- SET THE NODE FIELD VALUE TO USE BASED ON PROPERTY
		if (usenodeforsitenameparent = 1) then

			set nodetouse = new.Node;
		else

			set nodetouse = new.SiteName;
		end if;

		-- INSERT NEW SiteParent EVENT INTO alerts.status IF NEW OR UPDATED ScopeIDParent
		if (siteparentfound = 0 or newscopeparentneeded = 1 or scopeparentmoved = 1) then

			-- INSERT SYNTHETIC SiteNameParent EVENT
			insert into alerts.status (
				Identifier,
				Node,
				Class,
				Summary,
				AlertGroup,
				Severity,
				ScopeID,
				SiteName,
				FirstOccurrence,
				LastOccurrence,
				ParentIdentifier,
				Grade,
				OwnerUID,
				OwnerGID)
			values (
				siteidentifier,
				nodetouse,
				99990,
				new.SiteName + ': calculating summary details...',
				'SiteNameParent',
				1,
				scopeid,
				new.SiteName,
				now,
				now,
				scopeidentifier,
				1,
				65534,
				0);
		end if;

		-- LINK CURRENT EVENT TO SiteNameParent
		set new.ParentIdentifier = siteidentifier;
	end if;
end;
go

------------------------------------------------------------------------------
-- CREATE REINSERT TRIGGER ON alerts.status TO CREATE PARENT EVENTS AND LINK CHILDREN
------------------------------------------------------------------------------

CREATE OR REPLACE TRIGGER correlation_deduplication
GROUP correlation_triggers
PRIORITY 15
COMMENT 'Checks for the existence of parent events, updates or creates if necessary'
BEFORE REINSERT ON alerts.status
FOR EACH ROW
WHEN get_prop_value('ActingPrimary') %= 'TRUE' and
	(old.ScopeID != '' or new.ScopeID != '') and
	old.ParentIdentifier = '' and
	old.AlertGroup not in ('SiteNameParent', 'ScopeIDParent', 'Synthetic Event - Parent', 'ASMParent')

declare

	quietperiod		INTEGER;
	usenodeforscopeidparent	INTEGER;
	usenodeforsitenameparent INTEGER;
	nodetouse		CHAR(64);
	nositenameparentifsitenameblank INTEGER;

	now			TIME;
	expiretime		TIME;

	scopeid			CHAR(255);
	sitename		CHAR(255);

	scopeparentfound	INTEGER;
	newscopeparentneeded	INTEGER;
	scopeidentifier		CHAR(255);

	siteparentfound		INTEGER;
	scopeparentmoved	INTEGER;
	siteidentifier		CHAR(255);

begin

-- STEP 1: INITIALISE VARIABLES

	set quietperiod = 15 * 60;
	set usenodeforscopeidparent = 0;
	set usenodeforsitenameparent = 0;
	set nodetouse = '';
	set nositenameparentifsitenameblank = 0;

	set now = getdate();

	set scopeid = '';
	set sitename = '';

	set scopeparentfound = 0;
	set newscopeparentneeded = 0;
	set scopeidentifier = '';

	set siteparentfound = 0;
	set scopeparentmoved = 0;
	set siteidentifier = '';

	-- SET UP VARIABLES BASED ON PROPERTIES
	for each row property in master.properties where property.Name in (
		'SEGQuietPeriod',
		'SEGUseNodeForScopeIDParent',
		'SEGUseNodeForSiteNameParent',
		'SEGNoSiteNameParentIfSiteNameBlank')
	begin
		-- SPECIFIES THE GLOBAL DEFAULT QUIET PERIOD TO APPLY TO GROUPING
		if (property.Name = 'SEGQuietPeriod') then

			set quietperiod = property.IntValue;

		-- SPECIFIES IF THE NODE VALUE OF THE FIRST EVENT SHOULD BE USED FOR THE ScopeIDParent EVENT
		elseif (property.Name = 'SEGUseNodeForScopeIDParent') then

			set usenodeforscopeidparent = property.IntValue;

		-- SPECIFIES IF THE NODE VALUE OF THE FIRST EVENT SHOULD BE USED FOR THE SiteNameParent EVENT
		elseif (property.Name = 'SEGUseNodeForSiteNameParent') then

			set usenodeforsitenameparent = property.IntValue;

		-- SPECIFIES WHETHER OR NOT A SiteNameParent SHOULD BE CREATED IF SiteName IS NULL
		elseif (property.Name = 'SEGNoSiteNameParentIfSiteNameBlank') then

			set nositenameparentifsitenameblank = property.IntValue;

		end if;
	end;

	-- OVERRIDE QUIET PERIOD IF DEFINED IN THE INCOMING EVENT
	-- CHECK IF A PREVIOUS REINSERT TRIGGER HAS SET QuietPeriod
	if (old.QuietPeriod != 0) then

		set quietperiod = old.QuietPeriod;

	-- ELSE USE THE INCOMING VALUE
	elseif (new.QuietPeriod != 0) then

		set quietperiod = new.QuietPeriod;
	end if;

	-- USE VALUE STORED IN FirstOccurrence IF SET, ELSE USE CURRENT TIME
	if (old.FirstOccurrence = 0) then

		set now = getdate();
	else 

		set now = old.FirstOccurrence;
	end if;

	-- SET DEFAULT VALUE OF ExpireTime BASED ON FirstOccurrence
	set expiretime = now + quietperiod;

	-- SET THE ScopeID BASED ON THE INCOMING VALUE
	-- CHECK IF A PREVIOUS REINSERT TRIGGER HAS SET ScopeID
	if (old.ScopeID != '') then

		set scopeid = old.ScopeID;

	-- ELSE USE THE INCOMING VALUE
	elseif (new.ScopeID != '') then

		set scopeid = new.ScopeID;

		-- UPDATE ScopeID WITH INCOMING VALUE
		set old.ScopeID = new.ScopeID;
	end if;

	-- SET THE SiteName BASED ON THE INCOMING VALUE
	-- CHECK IF A PREVIOUS REINSERT TRIGGER HAS SET SiteName
	if (old.SiteName != '') then

		set sitename = old.SiteName;

	-- ELSE USE THE INCOMING VALUE
	elseif (new.SiteName != '') then

		set sitename = new.SiteName;

		-- UPDATE ScopeID WITH INCOMING VALUE
		set old.SiteName = new.SiteName;
	end if;


-- STEP 2: CHECK ScopeAlias MEMBERSHIP

	-- CHECK FOR ScopeAlias MEMBERSHIP
	for each row scopealiasmemberrow in master.correlation_scopealias_members where
		scopealiasmemberrow.ScopeID = scopeid
	begin

		-- SET THE ScopeID TO USE FOR THE SYNTHETIC CONTAINMENT EVENT TO BE THE ALIAS
		set scopeid = scopealiasmemberrow.ScopeAlias;
	end;

	-- USE INCOMING ScopeID VALUE IF ScopeID IS NOT A MEMBER OF AN ALIAS GROUP
--	if (scopeid = '') then
--
--		set scopeid = new.ScopeID;
--	end if;

-- STEP 3: SET UP ScopeIDParent EVENT

	-- LOOK FOR CURRENT ScopeIDParent
	for each row parent in master.correlation_scopeid where
		parent.ScopeID = scopeid
	begin

		-- MARK ScopeID PARENT AS FOUND
		set scopeparentfound = 1;

		-- CHECK IF THE GROUP ExpireTime HAS PASSED
		if (parent.ExpireTime < now) then

			-- IF SO, UPDATE PARENT ENTRY AND SET FLAG TO CREATE A NEW ScopeIDParent
			set parent.Identifier =	'SIDP:' + to_char(to_int(now)) + ':' + parent.ScopeID;
			set newscopeparentneeded = 1;
			set parent.ExpireTime = expiretime;
			set parent.CustomText =	'';
			set parent.CustomTimestamp = 0;
			set parent.CustomWeight = 0;
			set parent.HighImpactWeight = 0;
			set parent.HighImpactText = '';
			set parent.HighCauseWeight = 0;
			set parent.HighCauseText = '';
		end if;

		-- STORE EITHER EXISTING OR UPDATED ScopeIDParent IDENTIFIER
		set scopeidentifier = parent.Identifier;

		-- UPDATE LastOccurrence IN PARENT ENTRY IF INCOMING VALUE IS GREATER
		if (parent.LastOccurrence < now) then

			set parent.LastOccurrence = now;
		end if;

		-- UPDATE GROUP ExpireTime IF INCOMING ROW EXTENDS IT
		-- DO NOT EXTEND ExpireTime IF SCOPEID IS MARKED AS HAVING A FIXED TIME WINDOW
		-- FIXED TIME WINDOW SCOPEIDS ARE PREFIXED WITH THE STRING FX:
		if (parent.ExpireTime < expiretime and parent.ScopeID not like '^FX:') then

			set parent.ExpireTime = expiretime;
		end if;
	end;

	-- CREATE PARENT ENTRY IN master.correlation_scopeid IF NONE FOUND
	if (scopeparentfound = 0) then

		-- SET ScopeIDParent IDENTIFIER TO A NEW VALUE
		set scopeidentifier = 'SIDP:' + to_char(to_int(now)) + ':' + scopeid;

		-- INSERT NEW ScopeIDParent
		insert into master.correlation_scopeid (
			ScopeID,
			LastOccurrence,
			Identifier,
			ExpireTime)
		values (
			scopeid,
			now,
			scopeidentifier,
			expiretime);
	end if;

	-- SET THE NODE FIELD VALUE TO USE BASED ON PROPERTY
	if (usenodeforscopeidparent = 1) then

		set nodetouse = old.Node;
	else

		set nodetouse = scopeid;
	end if;

	-- INSERT ScopeIDParent EVENT INTO alerts.status IF NOT PRESENT OR NEW ONE NEEDED
	if (scopeparentfound = 0 or newscopeparentneeded = 1) then

		-- INSERT SYNTHETIC ScopeIDParent EVENT
		insert into alerts.status (
			Identifier,
			Node,
			Class,
			Summary,
			AlertGroup,
			Severity,
			ScopeID,
			FirstOccurrence,
			LastOccurrence,
			Grade,
			OwnerUID,
			OwnerGID)
		values (
			scopeidentifier,
			nodetouse,
			99990,
			'INCIDENT: ' + scopeid + ': calculating summary details...',
			'ScopeIDParent',
			1,
			scopeid,
			now,
			now,
			1,
			65534,
			0);
	end if;

-- STEP 4: SET UP SiteNameParent EVENT IF OPTION TO DISABLE IS NOT SET

	-- DO NOT CREATE SiteNameParent EVENT IF OPTION TO DISABLE SiteNameParent
	-- IS SET AND SiteName IS BLANK.  INSTEAD LINK EVENT TO ScopeIDParent EVENT
	if (nositenameparentifsitenameblank = 1 and sitename = '') then

		-- LINK CURRENT EVENT TO ScopeIDParent EVENT
		set old.ParentIdentifier = scopeidentifier;

	-- ELSE CREATE SiteNameParent EVENT AND LINK CURRENT EVENT TO THAT
	else

		-- IF EVENT HAS NO SiteName THEN SET IT TO A DEFAULT
		if (sitename = '') then

			set old.SiteName = 'NO SITENAME';
		end if;

		-- LOOK FOR CURRENT SiteNameParent
		for each row parent in master.correlation_sitename where
			parent.SiteName = sitename and
			parent.ScopeID = scopeid
		begin

			-- MARK SiteName PARENT AS FOUND
			set siteparentfound = 1;

			-- CHECK IF ScopeIDParent HAS MOVED
			for each row scopeidparentevent in alerts.status where
				scopeidparentevent.Identifier = parent.Identifier
			begin

				if (scopeidparentevent.ParentIdentifier != scopeidentifier) then

					set scopeparentmoved = 1;
				end if;
			end;

			-- UPDATE THE ENTRY IF WE ARE CREATING A NEW ScopeIDParent
			if (newscopeparentneeded = 1 or scopeparentmoved = 1) then

				set parent.Identifier = 'SNP:' + to_char(to_int(now)) + ':' + scopeid + ':' + sitename;
				set parent.CustomText =	'';
				set parent.CustomTimestamp = 0;
				set parent.CustomWeight = 0;
				set parent.HighImpactWeight = 0;
				set parent.HighImpactText = '';
				set parent.HighCauseWeight = 0;
				set parent.HighCauseText = '';
			end if;

			-- STORE CURRENT SiteName PARENT IDENTIFIER
			set siteidentifier = parent.Identifier;
		end;

		-- CREATE PARENT ENTRY IN master.correlation_sitename IF NONE FOUND
		if (siteparentfound = 0) then

			-- STORE NEW SiteName PARENT IDENTIFIER
			set siteidentifier = 'SNP:' +
				to_char(to_int(now)) + ':' + scopeid + ':' + sitename;

			-- CREATE THE NEW SiteParent EVENT
			insert into master.correlation_sitename (
				SiteName,
				ScopeID,
				Identifier)
			values (
				sitename,
				scopeid,
				siteidentifier);
		end if;

		-- SET THE NODE FIELD VALUE TO USE BASED ON PROPERTY
		if (usenodeforsitenameparent = 1) then

			set nodetouse = old.Node;
		else

			set nodetouse = sitename;
		end if;

		-- INSERT NEW SiteParent EVENT INTO alerts.status IF NEW OR UPDATED ScopeIDParent
		if (siteparentfound = 0 or newscopeparentneeded = 1 or scopeparentmoved = 1) then

			-- INSERT SYNTHETIC SiteNameParent EVENT
			insert into alerts.status (
				Identifier,
				Node,
				Class,
				Summary,
				AlertGroup,
				Severity,
				ScopeID,
				SiteName,
				FirstOccurrence,
				LastOccurrence,
				ParentIdentifier,
				Grade,
				OwnerUID,
				OwnerGID)
			values (
				siteidentifier,
				nodetouse,
				99990,
				sitename + ': calculating summary details...',
				'SiteNameParent',
				1,
				scopeid,
				sitename,
				now,
				now,
				scopeidentifier,
				1,
				65534,
				0);
		end if;

		-- LINK CURRENT EVENT TO SiteNameParent
		set old.ParentIdentifier = siteidentifier;
	end if;
end;
go

------------------------------------------------------------------------------
-- CREATE DELETE TRIGGER ON alerts.status TO REMOVE PARENT ENTRIES
------------------------------------------------------------------------------

CREATE OR REPLACE TRIGGER correlation_delete_row
GROUP correlation_triggers
PRIORITY 15
COMMENT 'Deletes from the master.correlation_* tables where needed'
AFTER DELETE ON alerts.status
FOR EACH ROW
WHEN get_prop_value('ActingPrimary') %= 'TRUE'
begin

	-- DELETE CORRESPONDING ENTRY IN MASTER CORRELATION TABLE FOR SiteNameParent EVENTS
	if (old.AlertGroup = 'SiteNameParent') then

		delete from master.correlation_sitename where Identifier = old.Identifier;

	-- DELETE CORRESPONDING ENTRY IN MASTER CORRELATION TABLE FOR ScopeIDParent EVENTS
	elseif (old.AlertGroup = 'ScopeIDParent') then

		delete from master.correlation_scopeid where Identifier = old.Identifier;

	end if;

	-- DELETE CORRESPONDING ENTRY IN MASTER CORRELATION TABLE FOR PRIORITY CHILD EVENTS
	delete from master.correlation_priority_children where Identifier = old.Identifier;
end;
go

------------------------------------------------------------------------------
-- CREATE A PROCEDURE TO CLEAR EXPIRED PARENT EVENTS
------------------------------------------------------------------------------

CREATE OR REPLACE PROCEDURE correlation_clear_expired_parents ( )
declare

	parentfound INTEGER;

begin

	-- INITIALISE VARIABLES
	set parentfound = 0;

	-- TARGET PARENT EVENTS THAT ARE NOT CLEAR AND THAT HAVE NO CHILDREN EVENTS
	for each row parent in alerts.status where
		parent.Severity != 0 and
		parent.AlertGroup in ('SiteNameParent', 'ScopeIDParent', 'Synthetic Event - Parent', 'ASMParent') and
		parent.Identifier not in (select ParentIdentifier from alerts.status)
	begin

		-- DEAL WITH ANALYTICS BASED PARENTS FIRST
		if (parent.AlertGroup in ('Synthetic Event - Parent', 'ASMParent')) then

			-- CLEAR THE CHILDLESS PARENT
			set parent.Severity = 0;

		-- DEAL WITH SCOPE BASED PARENTS NEXT
		else

			-- MARK GROUPING ENTRY AS NOT FOUND INITIALLY
			set parentfound = 0;

			-- LOOK IN THE master.correlation_scopeid TABLE TO SEE IF THE ScopeID
			-- GROUPING HAS EXPIRED.  IF IT IS NOT FOUND, IT HAS ALREADY EXPIRED.
			for each row scopeidentry in master.correlation_scopeid where
				scopeidentry.ScopeID = parent.ScopeID
			begin

				-- IF CURRENT EVENT IS A SiteNameParent EVENT, CHECK THAT THE ACTIVE ScopeIDParent
				-- PRESENT IS ACTUALLY ITS DIRECT PARENT.  IF IT IS, THEN MARK IT AS FOUND.
				if (parent.AlertGroup = 'SiteNameParent' and
					parent.ParentIdentifier = scopeidentry.Identifier) then

					set parentfound = 1;

					-- CHECK IF ScopeID HAS EXPIRED
					if (scopeidentry.ExpireTime < getdate()) then

						-- CLEAR PARENT EVENT DUE TO GROUP EXPIRATION
						set parent.Severity = 0;
					end if;

				-- ELSE CHECK IF THE CURRENT PARENT IS THE CURRENT ACTIVE GROUPING
				elseif (parent.Identifier = scopeidentry.Identifier) then

					-- MARK ScopeID ENTRY AS HAVING BEEN FOUND
					set parentfound = 1;

					-- CHECK IF ScopeID HAS EXPIRED
					if (scopeidentry.ExpireTime < getdate()) then

						-- CLEAR PARENT EVENT DUE TO GROUP EXPIRATION
						set parent.Severity = 0;
					end if;
				end if;
			end;

			-- IF ScopeID GROUPING IS NOT FOUND, CLEAR IT AS IT HAS ALREADY EXPIRED
			if (parentfound = 0) then

				set parent.Severity = 0;
			end if;
		end if;
	end;
end;
go

------------------------------------------------------------------------------
-- CREATE A PROCEDURE TO UPDATE THE STORED PRIORITY CHILD EVENT INFORMATION
------------------------------------------------------------------------------

CREATE OR REPLACE procedure correlation_update_priority_child
( in identifier char(255),		-- IDENTIFIER OF THE PARENT EVENT
  in propagatetexttoparentcause int,	-- PROPERTY TO PROPAGATE CustomText BASED ON HIGHEST CHILD CauseWeight
  in highcauseweight int,		-- VARIABLE THAT HOLDS HIGHEST CauseWeight OF CHILDREN
  in highcausetext char(255),		-- VARIABLE THAT HOLDS HIGHEST CauseText OF CHILDREN
  in propagatetexttoparentimpact int,	-- PROPERTY TO PROPAGATE CustomText BASED ON HIGHEST CHILD ImpactWeight
  in highimpactweight int,		-- VARIABLE THAT HOLDS HIGHEST ImpactWeight OF CHILDREN
  in highimpacttext char(255),		-- VARIABLE THAT HOLDS HIGHEST ImpactText OF CHILDREN
  in propagatetexttoparentfirst int,	-- PROPERTY TO PROPAGATE CustomText BASED ON LOWEST CHILD FirstOccurrence
  in lowfirstoccurrence int,		-- VARIABLE THAT HOLDS LOWEST FirstOccurrence OF CHILDREN
  in propagatetexttoparentlast int,	-- PROPERTY TO PROPAGATE CustomText BASED ON HIGHEST CHILD LastOccurrence
  in highlastoccurrence int,		-- VARIABLE THAT HOLDS HIGHEST LastOccurrence OF CHILDREN
  in customtext char(4096) )		-- VARIABLE THAT HOLDS CustomText FROM CURRENT CHILDREN
declare

	found INTEGER;
	customweighttouse INTEGER;
	customtimestamptouse TIME;

begin
--
-- Procedure updates priority child event data or inserts a new record if none found.
--
-- Called by: correlation_process_sitenameparents, correlation_process_scopeidparents
--
-- Usage:  EXECUTE correlation_update_priority_child (site.Identifier,
--					propagatetexttositenameparentcause, highcauseweight, highcausetext,
--					propagatetexttositenameparentimpact, highimpactweight, highimpacttext,
--					propagatetexttositenameparentfirst, lowfirstoccurrence,
--					propagatetexttositenameparentlast, highlastoccurrence,
--					customtext);
--

	-- INITIALISE VARIABLES
	set found = 0;
	set customweighttouse = 0;
	set customtimestamptouse = 0;

	-- LOOK FOR EXISTING RECORD
	-- STORE HIGHEST ImpactWeight AND HIGHEST CauseWeight
	-- STORE CustomText AND APPROPRIATE ASSOCIATED METRIC
	for each row parentrec in master.correlation_priority_children where parentrec.Identifier = identifier
	begin

		-- MARK RECORD AS FOUND
		set found = 1;

		-- UPDATE HIGHEST STORED CauseWeight IF NEW HIGHER VALUE FOUND
		if (highcauseweight > parentrec.HighCauseWeight) then

			set parentrec.HighCauseWeight = highcauseweight;
			set parentrec.HighCauseText = highcausetext;
		end if;

		-- UPDATE HIGHEST STORED ImpactWeight IF NEW HIGHER VALUE FOUND
		if (highimpactweight > parentrec.HighImpactWeight) then

			set parentrec.HighImpactWeight = highimpactweight;
			set parentrec.HighImpactText = highimpacttext;
		end if;

		-- ASSESS IF THE PRIORITY CustomText SHOULD BE UPDATED
		-- CONSIDER THE HIGH CAUSE WEIGHT
		if (propagatetexttoparentcause = 1) then

			if (highcauseweight > parentrec.CustomWeight) then

				-- UPDATE WITH THE CURRENT HIGH EVENT
				set parentrec.CustomText = customtext;
				set parentrec.CustomWeight = highcauseweight;
			end if;

		-- CONSIDER THE HIGH IMPACT WEIGHT
		elseif (propagatetexttoparentimpact = 1) then

			if (highimpactweight > parentrec.CustomWeight) then

				-- UPDATE WITH THE CURRENT HIGH EVENT
				set parentrec.CustomText = customtext;
				set parentrec.CustomWeight = highimpactweight;
			end if;

		-- CONSIDER THE FIRST FirstOccurrence
		elseif (propagatetexttoparentfirst = 1 and lowfirstoccurrence != 0) then

			if (lowfirstoccurrence < parentrec.CustomTimestamp or parentrec.CustomTimestamp = 0) then

				-- UPDATE WITH THE CURRENT HIGH EVENT
				set parentrec.CustomText = customtext;
				set parentrec.CustomTimestamp = lowfirstoccurrence;
			end if;

		-- CONSIDER THE LAST LastOccurrence
		elseif (propagatetexttoparentlast = 1) then

			if (highlastoccurrence > parentrec.CustomTimestamp or parentrec.CustomTimestamp = 0) then

				-- UPDATE WITH THE CURRENT HIGH EVENT
				set parentrec.CustomText = customtext;
				set parentrec.CustomTimestamp = highlastoccurrence;
			end if;
		end if;
	end;

	-- INSERT NEW RECORD IF IT IS NOT FOUND
	if (found = 0) then

		-- ASSESS WHAT THE PRIORITY CustomText SHOULD BE
		-- CONSIDER THE HIGH CAUSE WEIGHT
		if (propagatetexttoparentcause = 1) then

			-- UPDATE WITH THE CURRENT HIGH EVENT
			set customweighttouse = highcauseweight;

		-- CONSIDER THE HIGH IMPACT WEIGHT
		elseif (propagatetexttoparentimpact = 1) then

			-- UPDATE WITH THE CURRENT HIGH EVENT
			set customweighttouse = highimpactweight;

		-- CONSIDER THE FIRST FirstOccurrence
		elseif (propagatetexttoparentfirst = 1 and lowfirstoccurrence != 0) then

			set customtimestamptouse = lowfirstoccurrence;

		-- CONSIDER THE LAST LastOccurrence
		elseif (propagatetexttoparentlast = 1) then

			-- UPDATE WITH THE CURRENT HIGH EVENT
			set customtimestamptouse = highlastoccurrence;
		end if;

		-- CREATE PRIORITY CHILD RECORD FOR PARENT EVENT
		insert into master.correlation_priority_children (
			Identifier,
			HighImpactWeight,
			HighImpactText,
			HighCauseWeight,
			HighCauseText,
			CustomText,
			CustomWeight,
			CustomTimestamp)
		values (
			identifier,
			highimpactweight,
			highimpacttext,
			highcauseweight,
			highcausetext,
			customtext,
			customweighttouse,
			customtimestamptouse);
	end if;
end;
go

------------------------------------------------------------------------------
-- CREATE A PROCEDURE TO CONSTRUCT IMPACT CAUSE TEXT FOR THE PARENT EVENTS
------------------------------------------------------------------------------

CREATE OR REPLACE procedure correlation_construct_impactcause
( in identifier char(255),			-- IDENTIFIER OF THE SiteNameParent EVENT
  in out summary char(255) )			-- VARIABLE THAT HOLDS THE CONSTRUCTED summary SO FAR
begin
--
-- Procedure constructs the impact cause text in the summary for the parent event based on the priority child events.
--
-- Called by: correlation_process_sitenameparents, correlation_process_scopeidparents
--
-- Usage:  EXECUTE correlation_construct_impactcause (site.Identifier, summary);
--

	-- APPEND A COLON IF summary NOT BLANK
	if (summary != '') then
		set summary = summary + ': ';
	end if;

	-- FETCH THE CURRENT HIGHEST CAUSE AND IMPACT INFORMATION AND THEN
	-- BUILD THE IMPACT/CAUSE TEXT BASED ON THE HIGHEST WEIGHTED ITEMS
	for each row parentrec in master.correlation_priority_children where parentrec.Identifier = identifier
	begin

		-- NO CAUSE OR IMPACT ARE CURRENTLY PRESENT OR STORED
		if (parentrec.HighCauseText = 'UNKNOWN' and parentrec.HighImpactText = 'UNKNOWN') then

			set summary = summary + 'CAUSE AND IMPACT: UNKNOWN';

		-- THE IMPACT IS KNOWN BUT CAUSE IS NOT
		elseif (parentrec.HighCauseText = 'UNKNOWN' and parentrec.HighImpactText != 'UNKNOWN') then

			set summary = summary + parentrec.HighImpactText + ' UNKNOWN CAUSE';

		-- THE CAUSE IS KNOWN BUT IMPACT IS NOT
		elseif (parentrec.HighCauseText != 'UNKNOWN' and parentrec.HighImpactText = 'UNKNOWN') then

			set summary = summary + parentrec.HighCauseText + ' UNKNOWN IMPACT';

		-- THE HIGHEST CAUSE AND IMPACT ARE THE SAME PROBLEM
		elseif (parentrec.HighCauseText != 'UNKNOWN' and parentrec.HighCauseText = parentrec.HighImpactText) then

			set summary = summary + 'CAUSE AND IMPACT: ' + parentrec.HighCauseText;

		-- BOTH THE CAUSE AND IMPACT ARE KNOWN
		else

			set summary = summary + parentrec.HighImpactText + ' caused by ' + parentrec.HighCauseText;
		end if;
	end;
end;
go

------------------------------------------------------------------------------
-- CREATE A PROCEDURE TO PROCESS SiteNameParent EVENTS
------------------------------------------------------------------------------

CREATE OR REPLACE PROCEDURE correlation_process_sitenameparents ( )
declare

	propagatettnumber INTEGER;			-- PROPERTY FOR WHETHER OR NOT TO PROPAGATE TTNumber TO CHILD EVENTS
	propagateacknowledged INTEGER;			-- PROPERTY FOR WHETHER OR NOT TO PROPAGATE Acknowledged TO CHILD EVENTS
	propagateowneruid INTEGER;			-- PROPERTY FOR WHETHER OR NOT TO PROPAGATE OwnerUID TO CHILD EVENTS
	propagateownergid INTEGER;			-- PROPERTY FOR WHETHER OR NOT TO PROPAGATE OwnerGID TO CHILD EVENTS

	usesitenameprefix INTEGER;			-- PROPERTY FOR WHETHER OR NOT TO USE THE ScopeID PREFIX IN THE SUMMARY
	sitenameprefix CHAR(255);			-- PROPERTY FOR THE ScopeID PREFIX
	usesitenamelabel INTEGER;			-- PROPERTY FOR WHETHER OR NOT TO USE THE ScopeID IN THE SUMMARY
	usesitenameimpactcause INTEGER;			-- PROPERTY FOR WHETHER OR NOT TO USE THE IMPACT CAUSE TEXT IN THE SUMMARY
	usesitenamecustomtext INTEGER;			-- PROPERTY FOR WHETHER OR NOT TO USE THE CustomText IN THE SUMMARY
	propagatetexttositenameparentcause INTEGER;	-- PROPERTY FOR PROPAGATION OF PRIORITY CHILD DATA BASED ON CauseWeight
	propagatetexttositenameparentimpact INTEGER;	-- PROPERTY FOR PROPAGATION OF PRIORITY CHILD DATA BASED ON ImpactWeight
	propagatetexttositenameparentfirst INTEGER;	-- PROPERTY FOR PROPAGATION OF PRIORITY CHILD DATA BASED ON FirstOccurrence
	propagatetexttositenameparentlast INTEGER;	-- PROPERTY FOR PROPAGATION OF PRIORITY CHILD DATA BASED ON LastOccurrence
	customtext CHAR(4096);				-- VARIABLE TO STORE CustomText FROM PRIORITY CHILD
	usesitenamenumactivealarms INTEGER;		-- PROPERTY FOR WHETHER OR NOT TO SHOW NUMBER OF SITES AFFECTED IN THE SUMMARY

	highcauseweight INTEGER;			-- VARIABLE USED TO STORE CHILD EVENT HIGHEST CauseWeight
	highcausetext CHAR(255);			-- VARIABLE USED TO STORE CHILD EVENT HIGHEST CauseWeight CustomText
	highimpactweight INTEGER;			-- VARIABLE USED TO STORE CHILD EVENT HIGHEST ImpactWeight
	highimpacttext CHAR(255);			-- VARIABLE USED TO STORE CHILD EVENT HIGHEST ImpactWeight CustomText
	highseverity INTEGER;				-- VARIABLE USED TO STORE CHILD EVENT HIGHEST Severity
	lowfirstoccurrence TIME;			-- VARIABLE USED TO STORE CHILD EVENT LOWEST FirstOccurrence
	highlastoccurrence TIME;			-- VARIABLE USED TO STORE CHILD EVENT HIGHEST LastOccurrence
	alarmcounter INTEGER;				-- VARIABLE USED TO TALLY UP THE NUMBER OF SUB-GROUPS
	summary CHAR(255);				-- VARIABLE USED TO CONSTRUCT THE Summary FIELD FOR EACH ScopeIDParent EVENT

	journaltoscopeidparent INTEGER;			-- PROPERTY FOR WHETHER OR NOT TO JOURNAL TO ScopeIDParent EVENTS
	scopeidserial INTEGER;				-- VARIABLE USED TO STORE THE Serial OF THE CURRENT ScopeIDParent EVENT
	scopeidparentidentifier CHAR(255);		-- VARIABLE USED TO STORE THE Identifier OF THE CURRENT ScopeIDParent EVENT
	scopeidjournalcount INTEGER;			-- VARIABLE USED TO STORE THE COUNT OF EVENTS THAT HAVE BEEN JOURNALLED TO THE CURRENT ScopeIDParent EVENT

	journaltositenameparent INTEGER;		-- PROPERTY FOR WHETHER OR NOT TO JOURNAL TO ScopeIDParent EVENTS
	journalmaxeventsperentry INTEGER;		-- PROPERTY FOR WHAT THE MAXIMUM NUMBER OF EVENTS IS PER JOURNAL ENTRY
	journalservernameserverserial INTEGER;		-- PROPERTY FOR WHETHER OR NOT TO INCLUDE CHILD ServerName AND ServerSerial IN JOURNAL ENTRIES
	journalnode INTEGER;				-- PROPERTY FOR WHETHER OR NOT TO INCLUDE CHILD Node IN JOURNAL ENTRIES
	journalsummary INTEGER;				-- PROPERTY FOR WHETHER OR NOT TO INCLUDE CHILD Summary IN JOURNAL ENTRIES
	journalalertkey INTEGER;			-- PROPERTY FOR WHETHER OR NOT TO INCLUDE CHILD AlertKey IN JOURNAL ENTRIES
	journalcustomtext INTEGER;			-- PROPERTY FOR WHETHER OR NOT TO INCLUDE CHILD CustomText IN JOURNAL ENTRIES
	maxsitenamejournals INTEGER;			-- PROPERTY FOR THE MAXIMUM ALLOWABLE NUMBER OF CHILD EVENTS TO JOURNAL TO ScopeIDParent
	maxscopeidjournals INTEGER;			-- PROPERTY FOR THE MAXIMUM ALLOWABLE NUMBER OF CHILD EVENTS TO JOURNAL TO ScopeIDParent

	thisjournalentry CHAR(4080);			-- VARIABLE USED TO CONSTRUCT THE JOURNAL ENTRY FOR THE CURRENT EVENT
	sitenameparentjournalentry CHAR(4080);		-- VARIABLE USED TO STORE THE CUMULATIVE CHILD ENTRIES FOR EACH SiteNameParent JOURNAL ENTRY
	scopeidparentjournalentry CHAR(4080);		-- VARIABLE USED TO STORE THE CUMULATIVE CHILD ENTRIES FOR EACH ScopeIDParent JOURNAL ENTRY
	sitenamejournalcounter INTEGER;			-- VARIABLE USED TO COUNT THE NUMBER OF CHILD ENTRIES THAT HAVE BEEN ADDED TO THE CURRENT SiteNameParent JOURNAL ENTRY
	scopeidjournalcounter INTEGER;			-- VARIABLE USED TO COUNT THE NUMBER OF CHILD ENTRIES THAT HAVE BEEN ADDED TO THE CURRENT ScopeIDParent JOURNAL ENTRY
	sitenamejournaloffset INTEGER;			-- VARIABLE USED TO STORE THE CURRENT UID OFFSET TO USE WHEN WRITING SiteNameParent JOURNALS
	scopeidjournaloffset INTEGER;			-- VARIABLE USED TO STORE THE CURRENT UID OFFSET TO USE WHEN WRITING ScopeIDParent JOURNALS

	sitenamesummaryactivefirst INTEGER;		-- PROPERTY FOR WHETHER OR NOT TO SHOW NUMBER OF ACTIVE ALARMS AT BEGINNING OF SUMMARY
	prioritytext CHAR(255);				-- VARIABLE USED TO STORE PRIORITY CHILD EVENT CustomText DATA
	prioritytimestamp TIME;				-- VARIABLE USED TO STORE PRIORITY CHILD EVENT TIMESTAMP DATA
	priorityweight INTEGER;				-- VARIABLE USED TO STORE PRIORITY CHILD EVENT WEIGHTING DATA

begin
--
-- Procedure updates SiteNameParent events.
-- This procedures uses properties stored in master.properties to drive its behaviour.
--
-- Called by: correlation_process_existing_parents
--
-- Usage:  EXECUTE correlation_process_sitenameparents ( );
--

	-- INITIALISE LOCAL VARIABLES
	set propagatettnumber = 1;
	set propagateacknowledged = 1;
	set propagateowneruid = 1;
	set propagateownergid = 1;

	set usesitenameprefix = 0;
	set sitenameprefix = '';
	set usesitenamelabel = 1;
	set usesitenameimpactcause = 1;
	set usesitenamecustomtext = 0;
	set propagatetexttositenameparentcause = 0;
	set propagatetexttositenameparentimpact = 0;
	set propagatetexttositenameparentfirst = 0;
	set propagatetexttositenameparentlast = 0;
	set customtext = '';
	set usesitenamenumactivealarms = 1;

	set highcauseweight = 0;
	set highcausetext = 'UNKNOWN';
	set highimpactweight = 0;
	set highimpacttext  = 'UNKNOWN';
	set highseverity = 0;
	set lowfirstoccurrence = 0;
	set highlastoccurrence = 0;
	set alarmcounter  = 0;
	set summary = '';

	set journaltoscopeidparent = 0;
	set scopeidserial = 0;
	set scopeidjournalcount = 0;
	set scopeidparentidentifier = '';

	set journaltositenameparent = 0;
	set journalmaxeventsperentry = 20;
	set journalservernameserverserial = 1;
	set journalnode = 1;
	set journalsummary = 1;
	set journalalertkey = 0;
	set journalcustomtext = 0;
	set maxsitenamejournals = 10;
	set maxscopeidjournals = 50;
	set thisjournalentry = '';
	set sitenameparentjournalentry = '';
	set scopeidparentjournalentry = '';
	set sitenamejournalcounter = 0;
	set sitenamejournaloffset = 0;
	set scopeidjournaloffset = 0;

	set sitenamesummaryactivefirst = 0;

	-- LOAD UP VARIABLES BASED ON PROPERTIES
	for each row property in master.properties where property.Name in (
		'SEGPropagateTTNumber',
		'SEGPropagateAcknowledged',
		'SEGPropagateOwnerUID',
		'SEGPropagateOwnerGID',
		'SEGUseSiteNamePrefix',
		'SEGSiteNamePrefix',
		'SEGUseSiteNameLabel',
		'SEGUseSiteNameImpactCause',
		'SEGUseSiteNameCustomText',
		'SEGPropagateTextToSiteNameParentCause',
		'SEGPropagateTextToSiteNameParentImpact',
		'SEGPropagateTextToSiteNameParentFirst',
		'SEGPropagateTextToSiteNameParentLast',
		'SEGUseSiteNameNumActiveAlarms',
		'SEGJournalUID',
		'SEGJournalToSiteNameParent',
		'SEGJournalToScopeIDParent',
		'SEGJournalMaxEventsPerEntry',
		'SEGJournalServerNameServerSerial',
		'SEGJournalNode',
		'SEGJournalSummary',
		'SEGJournalAlertKey',
		'SEGJournalCustomText',
		'SEGMaxSiteNameJournals',
		'SEGMaxScopeIDJournals',
		'SEGSiteNameSummaryActiveFirst')
	begin

		-- SPECIFIES WHETHER OR NOT TO PROPAGATE THE TTNumber TO CHILD EVENTS
		if (property.Name = 'SEGPropagateTTNumber') then

			set propagatettnumber = property.IntValue;

		-- SPECIFIES WHETHER OR NOT TO PROPAGATE THE Acknowledged STATUS TO CHILD EVENTS
		elseif (property.Name = 'SEGPropagateAcknowledged') then

			set propagateacknowledged = property.IntValue;

		-- SPECIFIES WHETHER OR NOT TO PROPAGATE THE OwnerUID TO CHILD EVENTS
		elseif (property.Name = 'SEGPropagateOwnerUID') then

			set propagateowneruid = property.IntValue;

		-- SPECIFIES WHETHER OR NOT TO PROPAGATE THE OwnerGID TO CHILD EVENTS
		elseif (property.Name = 'SEGPropagateOwnerGID') then

			set propagateownergid = property.IntValue;

		-- SPECIFIES WHETHER OR NOT TO INCLUDE THE SiteName PREFIX IN THE SiteNameParent EVENT
		elseif (property.Name = 'SEGUseSiteNamePrefix') then

			set usesitenameprefix = property.IntValue;

		-- SPECIFIES THE SiteName PREFIX TO USE IN THE SiteNameParent EVENT
		elseif (property.Name = 'SEGSiteNamePrefix') then

			set sitenameprefix = property.CharValue;

		-- SPECIFIES WHETHER OR NOT TO INCLUDE THE ACTUAL SiteName IN THE SiteNameParent EVENT
		elseif (property.Name = 'SEGUseSiteNameLabel') then

			set usesitenamelabel = property.IntValue;

		-- SPECIFIES WHETHER OR NOT TO INCLUDE THE IMPACT AND CAUSE INFO IN THE SiteNameParent EVENT
		elseif (property.Name = 'SEGUseSiteNameImpactCause') then

			set usesitenameimpactcause = property.IntValue;

		-- SPECIFIES WHETHER OR NOT TO INCLUDE THE CUSTOM TEXT IN THE SiteNameParent EVENT
		elseif (property.Name = 'SEGUseSiteNameCustomText') then

			set usesitenamecustomtext = property.IntValue;

		-- SPECIFIES WHETHER OR NOT TO PROPAGATE THE CUSTOM TEXT TO THE ScopeIDParent EVENT
		-- BASED ON THE CUSTOM TEXT OF THE CHILD EVENT WITH THE HIGHEST CauseWeight
		elseif (property.Name = 'SEGPropagateTextToSiteNameParentCause') then

			set propagatetexttositenameparentcause = property.IntValue;

		-- SPECIFIES WHETHER OR NOT TO PROPAGATE THE CUSTOM TEXT TO THE ScopeIDParent EVENT
		-- BASED ON THE CUSTOM TEXT OF THE CHILD EVENT WITH THE HIGHEST ImpactWeight
		elseif (property.Name = 'SEGPropagateTextToSiteNameParentImpact') then

			set propagatetexttositenameparentimpact = property.IntValue;

		-- SPECIFIES WHETHER OR NOT TO PROPAGATE THE CUSTOM TEXT TO THE ScopeIDParent EVENT
		-- BASED ON THE CUSTOM TEXT OF THE CHILD EVENT WITH THE LOWEST FirstOccurrence
		elseif (property.Name = 'SEGPropagateTextToSiteNameParentFirst') then

			set propagatetexttositenameparentfirst = property.IntValue;

		-- SPECIFIES WHETHER OR NOT TO PROPAGATE THE CUSTOM TEXT TO THE ScopeIDParent EVENT
		-- BASED ON THE CUSTOM TEXT OF THE CHILD EVENT WITH THE HIGHEST LastOccurrence
		elseif (property.Name = 'SEGPropagateTextToSiteNameParentLast') then

			set propagatetexttositenameparentlast = property.IntValue;

		-- SPECIFIES WHETHER OR NOT TO INCLUDE THE NUMBER OF ACTIVE ALARMS IN THE SiteNameParent EVENT
		elseif (property.Name = 'SEGUseSiteNameNumActiveAlarms') then

			set usesitenamenumactivealarms = property.IntValue;

		-- SPECIFIES THE UID TO USE WHEN INSERTING JOURNALS INTO SYNTHETIC PARENT EVENTS - DEFAULT IS ROOT
--		elseif (property.Name = 'SEGJournalUID') then
--
--			set journaluid = property.IntValue;

		-- SPECIFIES WHETHER OR NOT TO JOURNAL CHILD EVENTS TO THE ScopeIDParent EVENT
		elseif (property.Name = 'SEGJournalToScopeIDParent') then

			set journaltoscopeidparent = property.IntValue;

		-- SPECIFIES WHETHER OR NOT TO JOURNAL CHILD EVENTS TO THE SiteNameParent EVENT
		elseif (property.Name = 'SEGJournalToSiteNameParent') then

			set journaltositenameparent = property.IntValue;

		-- SPECIFIES HOW MANY EVENTS TO ROLL UP INTO A SINGLE JOURNAL ENTRY
		elseif (property.Name = 'SEGJournalMaxEventsPerEntry') then

			set journalmaxeventsperentry = property.IntValue;

		-- SPECIFIES WHETHER OR NOT TO SEND THE ServerName:ServerSerial FIELDS TO THE PARENT EVENT JOURNAL
		elseif (property.Name = 'SEGJournalServerNameServerSerial') then

			set journalservernameserverserial = property.IntValue;

		-- SPECIFIES WHETHER OR NOT TO SEND THE Node FIELD TO THE PARENT EVENT JOURNAL
		elseif (property.Name = 'SEGJournalNode') then

			set journalnode = property.IntValue;

		-- SPECIFIES WHETHER OR NOT TO SEND THE Summary FIELD TO THE PARENT EVENT JOURNAL
		elseif (property.Name = 'SEGJournalSummary') then

			set journalsummary = property.IntValue;

		-- SPECIFIES WHETHER OR NOT TO SEND THE AlertKey FIELD TO THE PARENT EVENT JOURNAL
		elseif (property.Name = 'SEGJournalAlertKey') then

			set journalalertkey = property.IntValue;

		-- SPECIFIES WHETHER OR NOT TO SEND THE CustomText FIELD TO THE PARENT EVENT JOURNAL
		elseif (property.Name = 'SEGJournalCustomText') then

			set journalcustomtext = property.IntValue;

		-- SPECIFIES MAXIMUM NUMBER OF EVENTS TO SEND TO THE SiteNameParent EVENT JOURNAL
		elseif (property.Name = 'SEGMaxSiteNameJournals') then

			set maxsitenamejournals = property.IntValue;

		-- SPECIFIES MAXIMUM NUMBER OF EVENTS TO SEND TO THE ScopeIDParent EVENT JOURNAL
		elseif (property.Name = 'SEGMaxScopeIDJournals') then

			set maxscopeidjournals = property.IntValue;

		-- SPECIFIES WHETHER OR NOT THE NUMBER OF ACTIVE ALARMS IS TOWARDS THE FRONT OF THE SUMMARY
		elseif (property.Name = 'SEGSiteNameSummaryActiveFirst') then

			set sitenamesummaryactivefirst = property.IntValue;
		end if;
	end;

	-- IN CASE MULTIPLE CustomText PROPAGATION OPTIONS ARE INADVERTENTLY SELECTED,
	-- ENFORCE AN ORDER OF PRECEDENCE TO ENSURE UNDEFINED RESULTS ARE AVOIDED
	--
	-- ORDER OF PRECEDENCE IS:
	-- 1 PROPAGATE CustomText OF HIGHEST CauseWeight
	-- 2 PROPAGATE CustomText OF HIGHEST ImpactWeight
	-- 3 PROPAGATE CustomText OF LOWEST FirstOccurrence
	-- 4 PROPAGATE CustomText OF HIGHEST LastOccurrence
	--
	if (propagatetexttositenameparentcause = 1) then

		-- CLEAR OTHER SELECTED OPTIONS
		set propagatetexttositenameparentimpact = 0;
		set propagatetexttositenameparentfirst = 0;
		set propagatetexttositenameparentlast = 0;

	elseif (propagatetexttositenameparentimpact = 1) then

		-- CLEAR OTHER SELECTED OPTIONS
		set propagatetexttositenameparentfirst = 0;
		set propagatetexttositenameparentlast = 0;

	elseif (propagatetexttositenameparentfirst = 1) then

		-- CLEAR OTHER SELECTED OPTIONS
		set propagatetexttositenameparentlast = 0;
	end if;

	-- UPDATE EACH SiteNameParent EVENT
	for each row site in alerts.status where site.AlertGroup = 'SiteNameParent'
	begin

		-- RESET LOCAL VARIABLES
		set highcauseweight = 0;
		set highcausetext = 'UNKNOWN';
		set highimpactweight = 0;
		set highimpacttext  = 'UNKNOWN';
		set highseverity = 0;
		set lowfirstoccurrence = 0;
		set highlastoccurrence = 0;
		set customtext = '';
		set alarmcounter  = 0;
		set summary = '';

		set scopeidserial = 0;
		set scopeidjournalcount = 0;
		set scopeidparentidentifier = '';

		set thisjournalentry = '';
		set sitenameparentjournalentry = '';
		set scopeidparentjournalentry = '';
		set sitenamejournalcounter = 0;
		set scopeidjournalcounter = 0;
		set sitenamejournaloffset = 0;

		-- IF WE ARE SENDING JOURNALS TO THE ScopeIDParent, WE NEED ITS SERIAL
		-- AND CURRENT JOURNAL COUNT
		if (journaltoscopeidparent = 1) then

			-- LOOK UP ScopeIDParent
			for each row scopeidparent in alerts.status where
				scopeidparent.Identifier = site.ParentIdentifier
			begin

				set scopeidserial = scopeidparent.Serial;
				set scopeidjournalcount = scopeidparent.Poll;
				set scopeidparentidentifier = scopeidparent.ParentIdentifier;
			end;
		end if;

		-- EXAMINE EACH CHILD OF THE CURRENT SiteNameParent EVENT
		for each row child in alerts.status where child.ParentIdentifier = site.Identifier
		begin

			-- ONLY CONSIDER NON-CLEARED CHILD EVENTS
			if (child.Severity > 0) then

				-- INCREMENT A COUNTER FOR THE NUMBER OF CHILDREN
				set alarmcounter = alarmcounter + 1;

				-- STORE THE HIGHEST Severity OF THE CHILDREN EVENTS
				if (highseverity < child.Severity) then

					set highseverity = child.Severity;
				end if;

				-- STORE THE HIGHEST CauseWeight OF THE CHILDREN EVENTS
				-- AND ITS ASSOCIATED NormalisedAlarmName
				if (child.CauseWeight > highcauseweight) then
	
					set highcauseweight = child.CauseWeight;
					set highcausetext = child.NormalisedAlarmName;

					-- IF OPTION TO PROPAGATE CUSTOM TEXT TO SiteNameParent EVENT FROM HIGHEST
					-- CauseWeight CHILD IS ENABLED, STORE CURRENT EVENT CUSTOM TEXT
					if (propagatetexttositenameparentcause = 1) then

						set customtext = child.CustomText;
					end if;
				end if;

				-- STORE THE HIGHEST ImpactWeight OF THE CHILDREN EVENTS
				-- AND ITS ASSOCIATED NormalisedAlarmName
				if (child.ImpactWeight > highimpactweight) then
	
					set highimpactweight = child.ImpactWeight;
					set highimpacttext = child.NormalisedAlarmName;

					-- IF OPTION TO PROPAGATE CUSTOM TEXT TO SiteNameParent EVENT FROM HIGHEST
					-- ImpactWeight CHILD IS ENABLED, STORE CURRENT EVENT CUSTOM TEXT
					if (propagatetexttositenameparentimpact = 1) then

						set customtext = child.CustomText;
					end if;
				end if;

				-- STORE THE LOWEST NON-ZERO FirstOccurrence OF THE CHILDREN EVENTS
				if (lowfirstoccurrence > child.FirstOccurrence or lowfirstoccurrence = 0) then

					set lowfirstoccurrence = child.FirstOccurrence;

					-- IF OPTION TO PROPAGATE CUSTOM TEXT TO SiteNameParent EVENT FROM LOWEST
					-- FirstOccurrence CHILD IS ENABLED, STORE CURRENT EVENT CUSTOM TEXT
					if (propagatetexttositenameparentfirst = 1) then

						set customtext = child.CustomText;
					end if;
				end if;

				-- STORE THE HIGHEST LastOccurrence OF THE CHILDREN EVENTS
				if (highlastoccurrence < child.LastOccurrence) then
	
					set highlastoccurrence = child.LastOccurrence;

					-- IF OPTION TO PROPAGATE CUSTOM TEXT TO SiteNameParent EVENT FROM HIGHEST
					-- LastOccurrence CHILD IS ENABLED, STORE CURRENT EVENT CUSTOM TEXT
					if (propagatetexttositenameparentlast = 1) then

						set customtext = child.CustomText;
					end if;
				end if;

				-- PROPAGATE THE TICKET NUMBER FROM THE SiteNameParent EVENT TO
				-- THE CURRENT CHILD IF THE CURRENT CHILD IS UNTICKETED,
				-- UNSUPPRESSED AND THE OPTION TO PROPAGATE TICKET NUMBER IS ENABLED
				if (site.TTNumber != '' and child.TTNumber = '' and
					child.SuppressEscl != 4 and propagatettnumber = 1) then
	
					set child.TTNumber = site.TTNumber;
				end if;

				-- PROPAGATE THE ACKNOWLEDGED STATUS FROM THE SiteNameParent EVENT TO
				-- THE CURRENT CHILD IF THE CURRENT CHILD IS UNACKNOWLEDGED AND THE
				-- OPTION TO PROPAGATE ACKNOWLEDGED STATUS IS ENABLED
				if (site.Acknowledged = 1 and child.Acknowledged != 1 and
					propagateacknowledged = 1) then

					set child.Acknowledged = site.Acknowledged;
				end if;

				-- PROPAGATE THE OwnerUID FROM THE SiteNameParent EVENT TO THE
				-- CURRENT CHILD EVENT IF THE OWNER OF THE SiteNameParent EVENT
				-- IS DIFFERENT TO THAT OF THE CHILD EVENT AND THE
				-- OPTION TO PROPAGATE OwnerUID IS ENABLED
				-- A VALUE OF 2 WILL ONLY PROPAGATE OWNERSHIP IF IT IS OWNED BY
				-- UID = Nobody 65534
				if (site.OwnerUID != child.OwnerUID and (
					 propagateowneruid = 1 or
					(propagateowneruid = 2 and child.OwnerUID = 65534))) then

					set child.OwnerUID = site.OwnerUID;
				end if;

				-- PROPAGATE THE OwnerGID FROM THE SiteNameParent EVENT TO THE
				-- CURRENT CHILD EVENT IF THE GROUP OF THE SiteNameParent EVENT
				-- IS DIFFERENT TO THAT OF THE CHILD EVENT AND THE
				-- OPTION TO PROPAGATE OwnerGID IS ENABLED
				-- A VALUE OF 2 WILL ONLY PROPAGATE OWNERSHIP IF IT IS OWNED BY
				-- GID = Public 0
				if (site.OwnerGID != child.OwnerGID and (
					 propagateownergid = 1 or
					(propagateownergid = 2 and child.OwnerGID = 0))) then

					set child.OwnerGID = site.OwnerGID;
				end if;

				-- CLEAR thisjournalentry VARIABLE
				set thisjournalentry = '';

				-- CONSTRUCT A JOURNAL ENTRY FOR CURRENT EVENT IF
				-- - THE CURRENT EVENT HAS NOT ALREADY BEEN JOURNALED
				-- - IF THE MAXIMUM NUMBER OF EVENTS PER JOURNAL ENTRY HAS NOT BEEN EXCEEDED
				-- - IF SENDING CHILD EVENT DATA AS JOURNALS TO SiteNameParent IS ENABLED OR
				-- - IF SENDING CHILD EVENT DATA AS JOURNALS TO ScopeIDParent IS ENABLED
				if (child.JournalSent = 0 and
					sitenamejournalcounter < journalmaxeventsperentry and
					scopeidjournalcounter < journalmaxeventsperentry and
					(journaltositenameparent = 1 OR journaltoscopeidparent = 1)) then

					-- CHECK WHETHER OR NOT TO INCLUDE THE ServerName:ServerSerial IN THIS JOURNAL ENTRY
					if (journalservernameserverserial = 1) then

						set thisjournalentry = thisjournalentry + child.ServerName +
							':' + to_char(child.ServerSerial);
					end if;

					-- CHECK WHETHER OR NOT TO INCLUDE THE Node IN THIS JOURNAL ENTRY
					if (journalnode = 1 and child.Node != '') then

						-- ADD COLON BEFORE APPENDING TEXT TO THIS JOURNAL ENTRY
						if (thisjournalentry != '') then
							set thisjournalentry = thisjournalentry + ': ';
						end if;

						set thisjournalentry = thisjournalentry + child.Node;
					end if;

					-- CHECK WHETHER OR NOT TO INCLUDE THE Summary IN THIS JOURNAL ENTRY
					if (journalsummary = 1 and child.Summary != '') then

						-- ADD COLON BEFORE APPENDING TEXT TO THIS JOURNAL ENTRY
						if (thisjournalentry != '') then
							set thisjournalentry = thisjournalentry + ': ';
						end if;

						set thisjournalentry = thisjournalentry + child.Summary;
					end if;

					-- CHECK WHETHER OR NOT TO INCLUDE THE AlertKey IN THIS JOURNAL ENTRY
					if (journalalertkey = 1 and child.AlertKey != '') then

						-- ADD COLON BEFORE APPENDING TEXT TO THIS JOURNAL ENTRY
						if (thisjournalentry != '') then
							set thisjournalentry = thisjournalentry + ': ';
						end if;

						set thisjournalentry = thisjournalentry + child.AlertKey;
					end if;

					-- CHECK WHETHER OR NOT TO INCLUDE THE CustomText IN THIS JOURNAL ENTRY
					if (journalcustomtext = 1 and child.CustomText != '') then

						-- ADD COLON BEFORE APPENDING TEXT TO THIS JOURNAL ENTRY
						if (thisjournalentry != '') then
							set thisjournalentry = thisjournalentry + ': ';
						end if;

						set thisjournalentry = thisjournalentry + child.CustomText;
					end if;
				end if;

				-- ADD CURRENT EVENT TO SiteNameParent JOURNAL ENTRY IF:
				-- - IF SENDING CHILD EVENT DATA AS JOURNALS TO SiteNameParent IS ENABLED
				-- - IF THE MAXIMUM NUMBER OF EVENTS FOR THIS SiteNameParent HAS NOT BEEN EXCEEDED
				-- - THE CURRENT JOURNAL ENTRY IS NOT NULL
				if (journaltositenameparent = 1 and
					(site.Poll + sitenamejournalcounter) < maxsitenamejournals and
					thisjournalentry != '') then

					-- APPEND A NEW LINE CHARACTER IF THERE IS ALREADY TEXT TO ADD
					if (sitenameparentjournalentry != '') then
						set sitenameparentjournalentry = sitenameparentjournalentry + '\n';
					end if;

					-- ADD THE CURRENT JOURNAL ENTRY TO THE EXISTING
					set sitenameparentjournalentry = sitenameparentjournalentry + thisjournalentry;

					-- INCREMENT THE SiteNameParent JOURNAL COUNTER
					set sitenamejournalcounter = sitenamejournalcounter + 1;

					-- MARK CHILD EVENT AS HAVING BEEN SENT
					set child.JournalSent = 1;
 				end if;

				-- ADD A JOURNAL ENTRY TO THE CURRENT SiteNameParent NOW IF MAX EVENTS PER ENTRY IS REACHED, UPDATE COUNT, RESET VARS
				if (sitenamejournalcounter = journalmaxeventsperentry) then

					EXECUTE jinsert(site.Serial, 1100000 + sitenamejournaloffset, getdate,
						'CHILD EVENTS:\n' + sitenameparentjournalentry);

					-- UPDATE THE NUMBER OF CHILD EVENTS APPENDED TO THIS PARENT EVENT
					set site.Poll = site.Poll + sitenamejournalcounter;

					-- APPEND NOTE THAT NO FURTHER JOURNALS WILL BE APPENDED TO THIS SiteNameParent
					-- DUE TO THE MAXIMUM NUMBER HAVING BEEN REACHED
					if (site.Poll = maxsitenamejournals) then

						EXECUTE jinsert(site.Serial, 1100000 + sitenamejournaloffset, getdate + 1,
							'*** MAXIMUM OF ' + to_char(maxsitenamejournals) +
							' EVENTS HAVE BEEN JOURNALED TO THIS PARENT');
					end if;

					-- INCREMENT THE JOURNAL UID OFFSET
					set sitenamejournaloffset = sitenamejournaloffset + 1;

					-- RESET SOME VARIABLES
					set sitenameparentjournalentry = '';
					set sitenamejournalcounter = 0;
				end if;

				-- ADD CURRENT EVENT TO ScopeIDParent JOURNAL ENTRY IF:
				-- - IF SENDING CHILD EVENT DATA AS JOURNALS TO ScopeIDParent IS ENABLED
				-- - IF THE MAXIMUM NUMBER OF EVENTS FOR THIS ScopeIDParent HAS NOT BEEN EXCEEDED
				-- - THE CURRENT JOURNAL ENTRY IS NOT NULL
				-- - THE CURRENT EVENT IS NOT A SiteNameParent EVENT
				-- - THE CURRENT EVENT IS NOT A ScopeIDParent EVENT - POSSIBLE IF THIS IS A RE PARENT
				if (journaltoscopeidparent = 1 and
					(scopeidjournalcount + scopeidjournalcounter) < maxscopeidjournals and
					thisjournalentry != '' and
					child.AlertGroup != 'SiteNameParent' and
					child.AlertGroup != 'ScopeIDParent' and
					child.AlertGroup != 'ASMParent') then

					-- APPEND A NEW LINE CHARACTER IF THERE IS ALREADY TEXT TO ADD
					if (scopeidparentjournalentry != '') then
						set scopeidparentjournalentry = scopeidparentjournalentry + '\n';
					end if;

					-- ADD THE CURRENT JOURNAL ENTRY TO THE EXISTING
					set scopeidparentjournalentry = scopeidparentjournalentry + thisjournalentry;

					-- INCREMENT THE SiteNameParent JOURNAL COUNTER
					set scopeidjournalcounter = scopeidjournalcounter + 1;

					-- MARK CHILD EVENT AS HAVING BEEN JOURNALLED
					set child.JournalSent = 1;
 				end if;

				-- ADD A JOURNAL ENTRY TO THE CURRENT ScopeIDParent NOW IF MAX EVENTS PER ENTRY IS REACHED, UPDATE COUNT, RESET VARS
				if (scopeidjournalcounter = journalmaxeventsperentry) then

					EXECUTE jinsert(scopeidserial, 1100000 + scopeidjournaloffset, getdate,
						'SUB-GROUPING CHILD EVENTS: ' + site.SiteName + ':\n' + scopeidparentjournalentry);

					-- UPDATE THE NUMBER OF CHILD EVENTS APPENDED TO THIS PARENT EVENT
					set scopeidjournalcount = scopeidjournalcount + scopeidjournalcounter;
					update alerts.status set Poll = scopeidjournalcount where Identifier = site.ParentIdentifier;

					-- APPEND NOTE THAT NO FURTHER JOURNALS WILL BE APPENDED TO THIS ScopeIDParent
					-- DUE TO THE MAXIMUM NUMBER HAVING BEEN REACHED
					if (scopeidjournalcount = maxscopeidjournals) then

						EXECUTE jinsert(scopeidserial, 1100000 + scopeidjournaloffset, getdate + 1,
							'*** MAXIMUM OF ' + to_char(maxscopeidjournals) +
							' EVENTS HAVE BEEN JOURNALED TO THIS PARENT');
					end if;

					-- INCREMENT THE JOURNAL UID OFFSET
					set scopeidjournaloffset = scopeidjournaloffset + 1;

					-- RESET SOME VARIABLES
					set scopeidparentjournalentry = '';
					set scopeidjournalcounter = 0;
				end if;
			end if;
		end;

		-- STORE THE NUMBER OF ACTIVE CHILD EVENTS IN THE GRADE FIELD, IF NOT ZERO
		if (alarmcounter != 0 and site.Grade != alarmcounter) then

			set site.Grade = alarmcounter;

		-- ELSE THE COUNT OF ACTIVE CHILD EVENTS IS ZERO, SO RESET GRADE FIELD TO ZERO
		elseif (alarmcounter = 0 and site.Grade != 0) then

			set site.Grade = 0;
		end if;

		-- BEGIN THE CONSTRUCTION OF THE SiteNameParent Summary FIELD

		-- SPECIFY WHETHER TO INCLUDE THE SiteName PREFIX
		if (usesitenameprefix = 1) then

			set summary = sitenameprefix;
		end if;

		-- SPECIFY WHETHER TO INCLUDE THE NUMBER OF ACTIVE ALARMS AT THE BEGINNING OF THE SUMMARY
		if (sitenamesummaryactivefirst = 1) then

			-- APPEND A SPACE CHARACTER IF summary NOT BLANK
			if (summary != '') then
				set summary = summary + ': ';
			end if;

			if (alarmcounter = 1) then
				set summary = summary + '(' + to_char(alarmcounter) + ' active alarm)';
			else
				set summary = summary + '(' + to_char(alarmcounter) + ' active alarms)';
			end if;
		end if;

                -- SPECIFY WHETHER TO INCLUDE THE SiteName LABEL
                if (usesitenamelabel = 1) then

                        -- APPEND A COLON IF summary NOT BLANK
                        if (summary != '') then
                                set summary = summary + ': ';
                        end if;

			set summary = summary + site.SiteName;
		end if;

		-- UPDATE PRIORITY CHILD DATA FOR THIS PARENT EVENT
		EXECUTE correlation_update_priority_child (site.Identifier,
			propagatetexttositenameparentcause, highcauseweight, highcausetext,
			propagatetexttositenameparentimpact, highimpactweight, highimpacttext,
			propagatetexttositenameparentfirst, lowfirstoccurrence,
			propagatetexttositenameparentlast, highlastoccurrence,
			customtext);

		-- APPEND IMPACT CAUSE TEXT TO SUMMARY IF CONFIGURED
		if (usesitenameimpactcause = 1) then

			EXECUTE correlation_construct_impactcause (site.Identifier, summary);
		end if;

		-- FETCH AND SET THE PRIORITY CHILD CauseWeight AND ImpactWeight VALUES
		if (	site.Severity != 0 and
			(propagatetexttositenameparentcause = 1 or
			 propagatetexttositenameparentimpact = 1 or
			 propagatetexttositenameparentfirst = 1 or
			 propagatetexttositenameparentlast = 1 or
			 usesitenameimpactcause = 1)) then

			for each row parentrec in master.correlation_priority_children where parentrec.Identifier = site.Identifier
			begin

				-- PROPAGATE PRIORITY CauseWeight TO PARENT IF IT HAS CHANGED
				if (site.CauseWeight != parentrec.HighCauseWeight) then

					set site.CauseWeight = parentrec.HighCauseWeight;
				end if;

				-- PROPAGATE PRIORITY ImpactWeight TO PARENT IF IT HAS CHANGED
				if (site.ImpactWeight != parentrec.HighImpactWeight) then

					set site.ImpactWeight = parentrec.HighImpactWeight;
				end if;
			end;
		end if;

		-- FETCH AND SET THE PRIORITY CustomText IF ANY OF THE PROPAGATION OPTIONS ARE SELECTED
		if (	site.Severity != 0 and
			(propagatetexttositenameparentcause = 1 or
			 propagatetexttositenameparentimpact = 1 or
			 propagatetexttositenameparentfirst = 1 or
			 propagatetexttositenameparentlast = 1)) then

			for each row parentrec in master.correlation_priority_children where parentrec.Identifier = site.Identifier
			begin

				-- PROPAGATE PRIORITY CustomText TO PARENT IF IT HAS CHANGED
				if (site.CustomText != parentrec.CustomText) then

					set site.CustomText = parentrec.CustomText;
				end if;
			end;

			-- SPECIFY WHETHER TO INCLUDE THE CUSTOM SUMMARY TEXT CustomText
			if (usesitenamecustomtext = 1) then

				-- APPEND A COLON IF summary NOT BLANK
				if (summary != '') then
					set summary = summary + ': ';
				end if;

				-- APPEND CustomText
				set summary = summary + site.CustomText;
			end if;
		end if;

		-- SPECIFY WHETHER TO INCLUDE THE NUMBER OF ACTIVE ALARMS AT THE END OF THE SUMMARY
		if (usesitenamenumactivealarms = 1) then

			-- APPEND A SPACE CHARACTER IF summary NOT BLANK
			if (summary != '') then
				set summary = summary + ' ';
			end if;

			if (alarmcounter = 1) then
				set summary = summary + '(' + to_char(alarmcounter) + ' active alarm)';
			else
				set summary = summary + '(' + to_char(alarmcounter) + ' active alarms)';
			end if;
		end if;

		-- UPDATE THE SiteNameParent EVENT IF NOT CLEAR
		-- OR IF THE SUMMARY HAS CHANGED
		if (site.Severity != 0 or site.Summary != summary) then

			-- UPDATE Summary FIELD ONLY IF IT HAS CHANGED
			if (site.Summary != summary) then

				set site.Summary = summary;
			end if;

			-- UPDATE Severity WITH INDETERMINITE IF ALL ITS CHILDREN ARE CLEAR OR
			-- THE HIGHEST SEVERITY OF ITS CHILDREN
			if (highseverity = 0 and site.Severity != 1) then
				set site.Severity = 1;
			elseif (highseverity != 0 and site.Severity != highseverity) then
				set site.Severity = highseverity;
			end if;

			-- UPDATE FirstOccurrence WITH THE EARLIEST VALUE FROM ITS CHILDREN
			if (lowfirstoccurrence != 0 and site.FirstOccurrence != lowfirstoccurrence) then
				set site.FirstOccurrence = lowfirstoccurrence;
			end if;

			-- UPDATE LastOccurrence WITH THE LATEST VALUE FROM ITS CHILDREN
			if (highlastoccurrence != 0 and site.LastOccurrence != highlastoccurrence) then
				set site.LastOccurrence = highlastoccurrence;
			end if;
		end if;

		-- ADD A JOURNAL ENTRY TO THE CURRENT SiteNameParent IF APPLICABLE AND UPDATE COUNT
		if (journaltositenameparent = 1 and sitenameparentjournalentry != '') then

			EXECUTE jinsert(site.Serial, 1100000 + sitenamejournaloffset, getdate,
				'CHILD EVENTS:\n' + sitenameparentjournalentry);

			-- UPDATE THE NUMBER OF CHILD EVENTS APPENDED TO THIS PARENT EVENT
			set site.Poll = site.Poll + sitenamejournalcounter;

			-- APPEND NOTE THAT NO FURTHER JOURNALS WILL BE APPENDED TO THIS SiteNameParent
			-- DUE TO THE MAXIMUM NUMBER HAVING BEEN REACHED
			if (site.Poll = maxsitenamejournals) then

				EXECUTE jinsert(site.Serial, 1100000 + sitenamejournaloffset, getdate + 1,
					'*** MAXIMUM OF ' + to_char(maxsitenamejournals) +
					' EVENTS HAVE BEEN JOURNALED TO THIS PARENT');
			end if;

			-- INCREMENT THE JOURNAL UID OFFSET
			set sitenamejournaloffset = sitenamejournaloffset + 1;
		end if;

		-- ADD A JOURNAL ENTRY TO THE CURRENT ScopeIDParent IF APPLICABLE AND UPDATE COUNT
		if (journaltoscopeidparent = 1 and scopeidparentjournalentry != '') then

			EXECUTE jinsert(scopeidserial, 1100000 + scopeidjournaloffset, getdate,
				'SUB-GROUPING CHILD EVENTS: ' + site.SiteName + ':\n' + scopeidparentjournalentry);

			-- UPDATE THE NUMBER OF CHILD EVENTS APPENDED TO THIS PARENT EVENT
			set scopeidjournalcount = scopeidjournalcount + scopeidjournalcounter;
			update alerts.status set Poll = scopeidjournalcount where Identifier = site.ParentIdentifier;

			-- APPEND NOTE THAT NO FURTHER JOURNALS WILL BE APPENDED TO THIS ScopeIDParent
			-- DUE TO THE MAXIMUM NUMBER HAVING BEEN REACHED
			if (scopeidjournalcount = maxscopeidjournals) then

				EXECUTE jinsert(scopeidserial, 1100000 + scopeidjournaloffset, getdate + 1,
					'*** MAXIMUM OF ' + to_char(maxscopeidjournals) +
					' EVENTS HAVE BEEN JOURNALED TO THIS PARENT');
			end if;

			-- INCREMENT THE JOURNAL UID OFFSET
			set scopeidjournaloffset = scopeidjournaloffset + 1;
		end if;
	end;
end;
go

------------------------------------------------------------------------------
-- CREATE A PROCEDURE TO PROCESS ScopeIDParent EVENTS
------------------------------------------------------------------------------

CREATE OR REPLACE PROCEDURE correlation_process_scopeidparents ( )
declare

	propagatettnumber INTEGER;			-- PROPERTY FOR WHETHER OR NOT TO PROPAGATE TTNumber TO CHILD EVENTS
	propagateacknowledged INTEGER;			-- PROPERTY FOR WHETHER OR NOT TO PROPAGATE Acknowledged TO CHILD EVENTS
	propagateowneruid INTEGER;			-- PROPERTY FOR WHETHER OR NOT TO PROPAGATE OwnerUID TO CHILD EVENTS
	propagateownergid INTEGER;			-- PROPERTY FOR WHETHER OR NOT TO PROPAGATE OwnerGID TO CHILD EVENTS

	usescopeidprefix INTEGER;			-- PROPERTY FOR WHETHER OR NOT TO USE THE ScopeID PREFIX IN THE SUMMARY
	scopeidprefix CHAR(255);			-- PROPERTY FOR THE ScopeID PREFIX
	usescopeidlabel INTEGER;			-- PROPERTY FOR WHETHER OR NOT TO USE THE ScopeID IN THE SUMMARY
	usescopeidimpactcause INTEGER;			-- PROPERTY FOR WHETHER OR NOT TO USE THE IMPACT CAUSE TEXT IN THE SUMMARY
	usescopeidcustomtext INTEGER;			-- PROPERTY FOR WHETHER OR NOT TO USE THE CustomText IN THE SUMMARY
	propagatetexttoscopeidparentcause INTEGER;	-- PROPERTY FOR PROPAGATION OF PRIORITY CHILD DATA BASED ON CauseWeight
	propagatetexttoscopeidparentimpact INTEGER;	-- PROPERTY FOR PROPAGATION OF PRIORITY CHILD DATA BASED ON ImpactWeight
	propagatetexttoscopeidparentfirst INTEGER;	-- PROPERTY FOR PROPAGATION OF PRIORITY CHILD DATA BASED ON FirstOccurrence
	propagatetexttoscopeidparentlast INTEGER;	-- PROPERTY FOR PROPAGATION OF PRIORITY CHILD DATA BASED ON LastOccurrence
	customtext CHAR(4096);				-- VARIABLE TO STORE CustomText FROM PRIORITY CHILD
	usescopeidsitesaffected INTEGER;		-- PROPERTY FOR WHETHER OR NOT TO SHOW NUMBER OF SITES AFFECTED IN THE SUMMARY
	scopeidsitesaffectedlabel CHAR(255);		-- PROPERTY FOR WHAT THE site LABEL SHOULD BE
	usescopeidnumactivealarms INTEGER;		-- PROPERTY FOR WHETHER OR NOT TO SHOW NUMBER OF ACTIVE ALARMS AT END OF SUMMARY

	highcauseweight INTEGER;			-- VARIABLE USED TO STORE CHILD EVENT HIGHEST CauseWeight
	highcausetext CHAR(255);			-- VARIABLE USED TO STORE CHILD EVENT HIGHEST CauseWeight CustomText
	highimpactweight INTEGER;			-- VARIABLE USED TO STORE CHILD EVENT HIGHEST ImpactWeight
	highimpacttext CHAR(255);			-- VARIABLE USED TO STORE CHILD EVENT HIGHEST ImpactWeight CustomText
	highseverity INTEGER;				-- VARIABLE USED TO STORE CHILD EVENT HIGHEST Severity
	lowfirstoccurrence TIME;			-- VARIABLE USED TO STORE CHILD EVENT LOWEST FirstOccurrence
	highlastoccurrence TIME;			-- VARIABLE USED TO STORE CHILD EVENT HIGHEST LastOccurrence
	alarmcounter INTEGER;				-- VARIABLE USED TO TALLY UP THE NUMBER OF SUB-GROUPS
	alarmcounter2 INTEGER;				-- VARIABLE USED TO TALLY UP THE NUMBER OF GRANDCHILD EVENTS
	alarmcounter3 INTEGER;				-- VARIABLE USED TO TALLY UP THE NUMBER OF DIRECT CHILD EVENTS
	summary CHAR(255);				-- VARIABLE USED TO CONSTRUCT THE Summary FIELD FOR EACH ScopeIDParent EVENT

	journaltoscopeidparent INTEGER;			-- PROPERTY FOR WHETHER OR NOT TO JOURNAL TO ScopeIDParent EVENTS
	journalmaxeventsperentry INTEGER;		-- PROPERTY FOR WHAT THE MAXIMUM NUMBER OF EVENTS IS PER JOURNAL ENTRY
	journalservernameserverserial INTEGER;		-- PROPERTY FOR WHETHER OR NOT TO INCLUDE CHILD ServerName AND ServerSerial IN JOURNAL ENTRIES
	journalnode INTEGER;				-- PROPERTY FOR WHETHER OR NOT TO INCLUDE CHILD Node IN JOURNAL ENTRIES
	journalsummary INTEGER;				-- PROPERTY FOR WHETHER OR NOT TO INCLUDE CHILD Summary IN JOURNAL ENTRIES
	journalalertkey INTEGER;			-- PROPERTY FOR WHETHER OR NOT TO INCLUDE CHILD AlertKey IN JOURNAL ENTRIES
	journalcustomtext INTEGER;			-- PROPERTY FOR WHETHER OR NOT TO INCLUDE CHILD CustomText IN JOURNAL ENTRIES
	maxscopeidjournals INTEGER;			-- PROPERTY FOR THE MAXIMUM ALLOWABLE NUMBER OF CHILD EVENTS TO JOURNAL TO ScopeIDParent EVENTS
	thisjournalentry CHAR(4080);			-- VARIABLE USED TO CONSTRUCT THE JOURNAL ENTRY FOR THE CURRENT EVENT
	scopeidparentjournalentry CHAR(4080);		-- VARIABLE USED TO STORE THE CUMULATIVE CHILD ENTRIES FOR EACH JOURNAL ENTRY
	journalcounter INTEGER;				-- VARIABLE USED TO COUNT THE NUMBER OF CHILD ENTRIES THAT HAVE BEEN ADDED TO THE CURRENT JOURNAL ENTRY
	journaloffset INTEGER;				-- VARIABLE USED TO INCREMENT THE UID VALUE USED WHEN CREATING JOURNALS TO AVOID PRIMARY KEY CLASH

	scopeidsummaryactivefirst INTEGER;		-- PROPERTY FOR WHETHER OR NOT TO SHOW NUMBER OF ACTIVE ALARMS AT BEGINNING OF SUMMARY
	prioritytext CHAR(255);				-- VARIABLE USED TO STORE PRIORITY CHILD EVENT CustomText DATA
	prioritytimestamp TIME;				-- VARIABLE USED TO STORE PRIORITY CHILD EVENT TIMESTAMP DATA
	priorityweight INTEGER;				-- VARIABLE USED TO STORE PRIORITY CHILD EVENT WEIGHTING DATA

begin
--
-- Procedure updates ScopeIDParent and synthetic parent events created by NOI analytics.
-- This procedures uses properties stored in master.properties to drive its behaviour.
--
-- Called by: correlation_process_existing_parents
--
-- Usage:  EXECUTE correlation_process_scopeidparents ( );
--

	-- INITIALISE LOCAL VARIABLES
	set propagatettnumber = 1;
	set propagateacknowledged = 1;
	set propagateowneruid = 1;
	set propagateownergid = 1;

	set usescopeidprefix = 1;
	set scopeidprefix = '';
	set usescopeidlabel = 1;
	set usescopeidimpactcause = 0;
	set usescopeidcustomtext = 0;
	set propagatetexttoscopeidparentcause = 0;
	set propagatetexttoscopeidparentimpact = 0;
	set propagatetexttoscopeidparentfirst = 0;
	set propagatetexttoscopeidparentlast = 0;
	set customtext = '';
	set usescopeidsitesaffected = 1;
	set scopeidsitesaffectedlabel = 'site';
	set usescopeidnumactivealarms = 0;

	set highcauseweight = 0;
	set highcausetext = 'UNKNOWN';
	set highimpactweight = 0;
	set highimpacttext  = 'UNKNOWN';
	set highseverity = 0;
	set lowfirstoccurrence = 0;
	set highlastoccurrence = 0;
	set alarmcounter= 0;
	set alarmcounter2 = 0;
	set alarmcounter3 = 0;
	set summary = '';

	set journaltoscopeidparent = 0;
	set journalmaxeventsperentry = 20;
	set journalservernameserverserial = 1;
	set journalnode = 1;
	set journalsummary = 1;
	set journalalertkey = 0;
	set journalcustomtext = 0;
	set maxscopeidjournals = 50;
	set thisjournalentry = '';
	set scopeidparentjournalentry = '';
	set journalcounter = 0;
	set journaloffset = 0;

	set scopeidsummaryactivefirst = 0;

	-- LOAD UP VARIABLES BASED ON PROPERTIES
	for each row property in master.properties where property.Name in (
		'SEGPropagateTTNumber',
		'SEGPropagateAcknowledged',
		'SEGPropagateOwnerUID',
		'SEGPropagateOwnerGID',
		'SEGUseScopeIDPrefix',
		'SEGScopeIDPrefix',
		'SEGUseScopeIDLabel',
		'SEGUseScopeIDImpactCause',
		'SEGUseScopeIDCustomText',
		'SEGPropagateTextToScopeIDParentCause',
		'SEGPropagateTextToScopeIDParentImpact',
		'SEGPropagateTextToScopeIDParentFirst',
		'SEGPropagateTextToScopeIDParentLast',
		'SEGUseScopeIDSitesAffected',
		'SEGScopeIDSitesAffectedLabel',
		'SEGUseScopeIDNumActiveAlarms',
		'SEGJournalUID',
		'SEGJournalToScopeIDParent',
		'SEGJournalMaxEventsPerEntry',
		'SEGJournalServerNameServerSerial',
		'SEGJournalNode',
		'SEGJournalSummary',
		'SEGJournalAlertKey',
		'SEGJournalCustomText',
		'SEGMaxScopeIDJournals',
		'SEGScopeIDSummaryActiveFirst')
	begin

		-- SPECIFIES WHETHER OR NOT TO PROPAGATE THE TTNumber TO CHILD EVENTS
		if (property.Name = 'SEGPropagateTTNumber') then

			set propagatettnumber = property.IntValue;

		-- SPECIFIES WHETHER OR NOT TO PROPAGATE THE Acknowledged STATUS TO CHILD EVENTS
		elseif (property.Name = 'SEGPropagateAcknowledged') then

			set propagateacknowledged = property.IntValue;

		-- SPECIFIES WHETHER OR NOT TO PROPAGATE THE OwnerUID TO CHILD EVENTS
		elseif (property.Name = 'SEGPropagateOwnerUID') then

			set propagateowneruid = property.IntValue;

		-- SPECIFIES WHETHER OR NOT TO PROPAGATE THE OwnerGID TO CHILD EVENTS
		elseif (property.Name = 'SEGPropagateOwnerGID') then

			set propagateownergid = property.IntValue;

		-- SPECIFIES WHETHER OR NOT TO USE THE ScopeID PREFIX IN THE ScopeIDParent EVENT
		elseif (property.Name = 'SEGUseScopeIDPrefix') then

			set usescopeidprefix = property.IntValue;

		-- SPECIFIES THE ScopeID PREFIX TO USE IN THE ScopeIDParent EVENT
		elseif (property.Name = 'SEGScopeIDPrefix') then

			set scopeidprefix = property.CharValue;

		-- SPECIFIES WHETHER OR NOT TO INCLUDE THE ScopeID LABEL IN THE ScopeIDParent EVENT
		elseif (property.Name = 'SEGUseScopeIDLabel') then

			set usescopeidlabel = property.IntValue;

		-- SPECIFIES WHETHER OR NOT TO INCLUDE THE IMPACT AND CAUSE INFO IN THE ScopeIDParent EVENT
		elseif (property.Name = 'SEGUseScopeIDImpactCause') then

			set usescopeidimpactcause = property.IntValue;

		-- SPECIFIES WHETHER OR NOT TO INCLUDE THE CUSTOM TEXT IN THE ScopeIDParent EVENT
		elseif (property.Name = 'SEGUseScopeIDCustomText') then

			set usescopeidcustomtext = property.IntValue;

		-- SPECIFIES WHETHER OR NOT TO PROPAGATE THE CUSTOM TEXT TO THE ScopeIDParent EVENT
		-- BASED ON THE CUSTOM TEXT OF THE CHILD EVENT WITH THE HIGHEST CauseWeight
		elseif (property.Name = 'SEGPropagateTextToScopeIDParentCause') then

			set propagatetexttoscopeidparentcause = property.IntValue;

		-- SPECIFIES WHETHER OR NOT TO PROPAGATE THE CUSTOM TEXT TO THE ScopeIDParent EVENT
		-- BASED ON THE CUSTOM TEXT OF THE CHILD EVENT WITH THE HIGHEST ImpactWeight
		elseif (property.Name = 'SEGPropagateTextToScopeIDParentImpact') then

			set propagatetexttoscopeidparentimpact = property.IntValue;

		-- SPECIFIES WHETHER OR NOT TO PROPAGATE THE CUSTOM TEXT TO THE ScopeIDParent EVENT
		-- BASED ON THE CUSTOM TEXT OF THE CHILD EVENT WITH THE LOWEST FirstOccurrence
		elseif (property.Name = 'SEGPropagateTextToScopeIDParentFirst') then

			set propagatetexttoscopeidparentfirst = property.IntValue;

		-- SPECIFIES WHETHER OR NOT TO PROPAGATE THE CUSTOM TEXT TO THE ScopeIDParent EVENT
		-- BASED ON THE CUSTOM TEXT OF THE CHILD EVENT WITH THE HIGHEST LastOccurrence
		elseif (property.Name = 'SEGPropagateTextToScopeIDParentLast') then

			set propagatetexttoscopeidparentlast = property.IntValue;

		-- SPECIFIES WHETHER OR NOT TO INCLUDE THE NUMBER OF AFFECTED SITES IN THE ScopeIDParent EVENT
		elseif (property.Name = 'SEGUseScopeIDSitesAffected') then

			set usescopeidsitesaffected = property.IntValue;

		-- SPECIFIES THE LABEL TO USE WHEN REFERRING TO THE NUMBER OF SUB-GROUPS AFFECTED IN THE ScopeIDParent SUMMARY FIELD
		elseif (property.Name = 'SEGScopeIDSitesAffectedLabel') then

			set scopeidsitesaffectedlabel = property.CharValue;

		-- SPECIFIES WHETHER OR NOT TO INCLUDE THE NUMBER OF ACTIVE ALARMS IN THE ScopeIDParent EVENT
		elseif (property.Name = 'SEGUseScopeIDNumActiveAlarms') then

			set usescopeidnumactivealarms = property.IntValue;

		-- SPECIFIES THE UID TO USE WHEN INSERTING JOURNALS INTO SYNTHETIC PARENT EVENTS - DEFAULT IS ROOT
--		elseif (property.Name = 'SEGJournalUID') then
--
--			set journaluid = property.IntValue;

		-- SPECIFIES WHETHER OR NOT TO JOURNAL CHILD EVENTS TO THE ScopeIDParent EVENT
		elseif (property.Name = 'SEGJournalToScopeIDParent') then

			set journaltoscopeidparent = property.IntValue;

		-- SPECIFIES HOW MANY EVENTS TO ROLL UP INTO A SINGLE JOURNAL ENTRY
		elseif (property.Name = 'SEGJournalMaxEventsPerEntry') then

			set journalmaxeventsperentry = property.IntValue;

		-- SPECIFIES WHETHER OR NOT TO SEND THE ServerName:ServerSerial FIELD TO THE PARENT EVENT JOURNAL
		elseif (property.Name = 'SEGJournalServerNameServerSerial') then

			set journalservernameserverserial = property.IntValue;

		-- SPECIFIES WHETHER OR NOT TO SEND THE Node FIELD TO THE PARENT EVENT JOURNAL
		elseif (property.Name = 'SEGJournalNode') then

			set journalnode = property.IntValue;

		-- SPECIFIES WHETHER OR NOT TO SEND THE Summary FIELD TO THE PARENT EVENT JOURNAL
		elseif (property.Name = 'SEGJournalSummary') then

			set journalsummary = property.IntValue;

		-- SPECIFIES WHETHER OR NOT TO SEND THE AlertKey FIELD TO THE PARENT EVENT JOURNAL
		elseif (property.Name = 'SEGJournalAlertKey') then

			set journalalertkey = property.IntValue;

		-- SPECIFIES WHETHER OR NOT TO SEND THE CustomText FIELD TO THE PARENT EVENT JOURNAL
		elseif (property.Name = 'SEGJournalCustomText') then

			set journalcustomtext = property.IntValue;

		-- SPECIFIES MAXIMUM NUMBER OF EVENTS TO SEND TO THE PARENT EVENT JOURNAL
		elseif (property.Name = 'SEGMaxScopeIDJournals') then

			set maxscopeidjournals = property.IntValue;

		-- SPECIFIES WHETHER OR NOT THE NUMBER OF ACTIVE ALARMS IS TOWARDS THE FRONT OF THE SUMMARY
		elseif (property.Name = 'SEGScopeIDSummaryActiveFirst') then

			set scopeidsummaryactivefirst = property.IntValue;
		end if;
	end;

	-- IN CASE MULTIPLE CustomText PROPAGATION OPTIONS ARE INADVERTENTLY SELECTED,
	-- ENFORCE AN ORDER OF PRECEDENCE TO ENSURE UNDEFINED RESULTS ARE AVOIDED
	--
	-- ORDER OF PRECEDENCE IS:
	-- 1 PROPAGATE CustomText OF HIGHEST CauseWeight
	-- 2 PROPAGATE CustomText OF HIGHEST ImpactWeight
	-- 3 PROPAGATE CustomText OF LOWEST FirstOccurrence
	-- 4 PROPAGATE CustomText OF HIGHEST LastOccurrence
	--
	if (propagatetexttoscopeidparentcause = 1) then

		-- CLEAR OTHER SELECTED OPTIONS
		set propagatetexttoscopeidparentimpact = 0;
		set propagatetexttoscopeidparentfirst = 0;
		set propagatetexttoscopeidparentlast = 0;

	elseif (propagatetexttoscopeidparentimpact = 1) then

		-- CLEAR OTHER SELECTED OPTIONS
		set propagatetexttoscopeidparentfirst = 0;
		set propagatetexttoscopeidparentlast = 0;

	elseif (propagatetexttoscopeidparentfirst = 1) then

		-- CLEAR OTHER SELECTED OPTIONS
		set propagatetexttoscopeidparentlast = 0;

	end if;

	-- UPDATE EACH ScopeIDParent EVENT
	for each row scopeid in alerts.status where scopeid.AlertGroup in ('ScopeIDParent', 'Synthetic Event - Parent', 'ASMParent', 'CustomParent')
	begin

		-- RESET LOCAL VARIABLES
		-- alarmcounter: USED TO COUNT UP NUMBER OF AFFECTED SITES
		-- alarmcounter2: USED TO COUNT UP NUMBER OF SITE EVENT CHILDREN
		-- alarmcounter3: USED TO COUNT UP NUMBER OF EVENTS DIRECTLY UNDER ScopeIDParent
		set highcauseweight = 0;
		set highcausetext = 'UNKNOWN';
		set highimpactweight = 0;
		set highimpacttext  = 'UNKNOWN';
		set highseverity = 0;
		set lowfirstoccurrence = 0;
		set highlastoccurrence = 0;
		set customtext = '';
		set alarmcounter  = 0;
		set alarmcounter2  = 0;
		set alarmcounter3  = 0;
		set summary = '';
		set thisjournalentry = '';
		set scopeidparentjournalentry = '';
		set journalcounter = 0;
		set journaloffset = 0;

		-- EXAMINE EACH CHILD OF THE CURRENT ScopeIDParent EVENT
		for each row child in alerts.status where child.ParentIdentifier = scopeid.Identifier
		begin

			-- ONLY CONSIDER NON-CLEARED CHILD EVENTS
			if (child.Severity > 0) then

				-- INCLUDE CURRENT CHILD EVENT IN COUNTS
				-- PROCESS SiteNameParent EVENTS
				if (child.AlertGroup in ('SiteNameParent', 'ScopeIDParent')) then

					-- EXCLUDE SiteNameParent EVENTS THAT HAVE NO SiteName
					if (child.AlertGroup = 'SiteNameParent' and child.SiteName != 'NO SITENAME') then

						-- INCREMENT NUMBER OF AFFECTED SITES
						set alarmcounter = alarmcounter + 1;
					end if;

					-- ADD TO EVENT COUNTER THE NUMBER OF UNDERLYING EVENTS
					set alarmcounter2 = alarmcounter2 + child.Grade;

					-- FETCH STORED PRIORITY CHILD INFORMATION
					for each row parentrec in master.correlation_priority_children where parentrec.Identifier = child.Identifier
					begin

						-- STORE THE HIGHEST CauseWeight OF THE CHILDREN EVENTS
						-- AND ITS ASSOCIATED NormalisedAlarmName
						if (child.CauseWeight > highcauseweight) then
	
							set highcauseweight = child.CauseWeight;
							set highcausetext = parentrec.HighCauseText;

							-- IF OPTION TO PROPAGATE CUSTOM TEXT TO ScopeIDParent EVENT FROM HIGHEST
							-- CauseWeight CHILD IS ENABLED, STORE CURRENT EVENT CUSTOM TEXT
							if (propagatetexttoscopeidparentcause = 1) then

								set customtext = child.CustomText;
							end if;
						end if;

						-- STORE THE HIGHEST ImpactWeight OF THE CHILDREN EVENTS
						-- AND ITS ASSOCIATED NormalisedAlarmName
						if (child.ImpactWeight > highimpactweight) then
	
							set highimpactweight = child.ImpactWeight;
							set highimpacttext = parentrec.HighImpactText;

							-- IF OPTION TO PROPAGATE CUSTOM TEXT TO ScopeIDParent EVENT FROM HIGHEST
							-- ImpactWeight CHILD IS ENABLED, STORE CURRENT EVENT CUSTOM TEXT
							if (propagatetexttoscopeidparentimpact = 1) then

								set customtext = child.CustomText;
							end if;
						end if;
					end;

				-- ELSE EVENT IS A REAL EVENT AND NOT A SYNTHETIC ONE
				else

					-- STORE THE HIGHEST CauseWeight OF THE CHILDREN EVENTS
					-- AND ITS ASSOCIATED NormalisedAlarmName
					if (child.CauseWeight > highcauseweight) then
	
						set highcauseweight = child.CauseWeight;
						set highcausetext = child.NormalisedAlarmName;

						-- IF OPTION TO PROPAGATE CUSTOM TEXT TO ScopeIDParent EVENT FROM HIGHEST
						-- CauseWeight CHILD IS ENABLED, STORE CURRENT EVENT CUSTOM TEXT
						if (propagatetexttoscopeidparentcause = 1) then

							set customtext = child.CustomText;
						end if;
					end if;

					-- STORE THE HIGHEST ImpactWeight OF THE CHILDREN EVENTS
					-- AND ITS ASSOCIATED NormalisedAlarmName
					if (child.ImpactWeight > highimpactweight) then
	
						set highimpactweight = child.ImpactWeight;
						set highimpacttext = child.NormalisedAlarmName;

						-- IF OPTION TO PROPAGATE CUSTOM TEXT TO ScopeIDParent EVENT FROM HIGHEST
						-- ImpactWeight CHILD IS ENABLED, STORE CURRENT EVENT CUSTOM TEXT
						if (propagatetexttoscopeidparentimpact = 1) then

							set customtext = child.CustomText;
						end if;
					end if;

					-- INCREMENT EVENT COUNTER
					set alarmcounter3 = alarmcounter3 + 1;
				end if;

				-- STORE THE HIGHEST Severity OF THE CHILDREN EVENTS
				if (highseverity < child.Severity) then

					set highseverity = child.Severity;
				end if;

				-- STORE THE LOWEST NON-ZERO FirstOccurrence OF THE CHILDREN EVENTS
				if (lowfirstoccurrence > child.FirstOccurrence or lowfirstoccurrence = 0) then

					set lowfirstoccurrence = child.FirstOccurrence;

					-- IF OPTION TO PROPAGATE CUSTOM TEXT TO ScopeIDParent EVENT FROM LOWEST
					-- FirstOccurrence CHILD IS ENABLED, STORE CURRENT EVENT CUSTOM TEXT
					if (propagatetexttoscopeidparentfirst = 1) then

						set customtext = child.CustomText;
					end if;
				end if;

				-- STORE THE HIGHEST LastOccurrence OF THE CHILDREN EVENTS
				if (highlastoccurrence < child.LastOccurrence) then

					set highlastoccurrence = child.LastOccurrence;

					-- IF OPTION TO PROPAGATE CUSTOM TEXT TO ScopeIDParent EVENT FROM HIGHEST
					-- LastOccurrence CHILD IS ENABLED, STORE CURRENT EVENT CUSTOM TEXT
					if (propagatetexttoscopeidparentlast = 1) then

						set customtext = child.CustomText;
					end if;
				end if;

				-- PROPAGATE THE TICKET NUMBER FROM THE ScopeIDParent EVENT TO
				-- THE CURRENT CHILD IF THE CURRENT CHILD IS UNTICKETED,
				-- UNSUPPRESSED AND THE OPTION TO PROPAGATE TICKET NUMBER IS ENABLED
				if (scopeid.TTNumber != '' and child.TTNumber = '' and
					child.SuppressEscl != 4 and propagatettnumber = 1) then

					set child.TTNumber = scopeid.TTNumber;
				end if;

				-- PROPAGATE THE ACKNOWLEDGED STATUS FROM THE ScopeIDParent EVENT TO
				-- THE CURRENT CHILD IF THE CURRENT CHILD IS UNACKNOWLEDGED AND THE
				-- OPTION TO PROPAGATE ACKNOWLEDGED STATUS IS ENABLED
				if (scopeid.Acknowledged = 1 and child.Acknowledged != 1
					and propagateacknowledged = 1) then

					set child.Acknowledged = scopeid.Acknowledged;
				end if;

				-- PROPAGATE THE OwnerUID FROM THE ScopeIDParent EVENT TO THE
				-- CURRENT CHILD EVENT IF THE OWNER OF THE ScopeIDParent EVENT
				-- IS DIFFERENT TO THAT OF THE CHILD EVENT AND THE
				-- OPTION TO PROPAGATE OwnerUID IS ENABLED
				-- A VALUE OF 2 WILL ONLY PROPAGATE OWNERSHIP IF IT IS OWNED BY
				-- UID = Nobody 65534
				if (scopeid.OwnerUID != child.OwnerUID and (
					 propagateowneruid = 1 or
					(propagateowneruid = 2 and child.OwnerUID = 65534))) then

					set child.OwnerUID = scopeid.OwnerUID;
				end if;

				-- PROPAGATE THE OwnerGID FROM THE ScopeIDParent EVENT TO THE
				-- CURRENT CHILD EVENT IF THE GROUP OF THE ScopeIDParent EVENT
				-- IS DIFFERENT TO THAT OF THE CHILD EVENT AND THE
				-- OPTION TO PROPAGATE OwnerGID IS ENABLED
				-- A VALUE OF 2 WILL ONLY PROPAGATE OWNERSHIP IF IT IS OWNED BY
				-- GID = Public 0
				if (scopeid.OwnerGID != child.OwnerGID and (
					 propagateownergid = 1 or
					(propagateownergid = 2 and child.OwnerGID = 0))) then

					set child.OwnerGID = scopeid.OwnerGID;
				end if;

				-- CLEAR thisjournalentry VARIABLE
				set thisjournalentry = '';

				-- CONSTRUCT A JOURNAL ENTRY FOR CURRENT EVENT IF
				-- - THE CURRENT EVENT HAS NOT ALREADY BEEN JOURNALED
				-- - IF SENDING CHILD EVENT DATA AS JOURNALS TO ScopeIDParent IS ENABLED
				-- - IF THE MAXIMUM NUMBER OF EVENTS PER JOURNAL ENTRY HAS NOT BEEN EXCEEDED
				-- - IF THE MAXIMUM NUMBER OF EVENTS FOR THIS ScopeIDParent HAS NOT BEEN EXCEEDED
				-- - THE CURRENT EVENT IS NOT A SiteNameParent EVENT
				-- - THE CURRENT EVENT IS NOT A ScopeIDParent EVENT - POSSIBLE IF THIS IS A RE PARENT
				if (child.JournalSent = 0 and journaltoscopeidparent = 1 and
					journalcounter < journalmaxeventsperentry and
					(scopeid.Poll + journalcounter) < maxscopeidjournals and
					child.AlertGroup != 'SiteNameParent' and
					child.AlertGroup != 'ScopeIDParent' and
					child.AlertGroup != 'ASMParent') then

					-- CHECK WHETHER OR NOT TO INCLUDE THE ServerName:ServerSerial IN THIS JOURNAL ENTRY
					if (journalservernameserverserial = 1) then

						set thisjournalentry = thisjournalentry + child.ServerName +
							':' + to_char(child.ServerSerial);
					end if;

					-- CHECK WHETHER OR NOT TO INCLUDE THE Node IN THIS JOURNAL ENTRY
					if (journalnode = 1 and child.Node != '') then

						-- ADD COLON BEFORE APPENDING TEXT TO THIS JOURNAL ENTRY
						if (thisjournalentry != '') then
							set thisjournalentry = thisjournalentry + ': ';
						end if;

						set thisjournalentry = thisjournalentry + child.Node;
					end if;

					-- CHECK WHETHER OR NOT TO INCLUDE THE Summary IN THIS JOURNAL ENTRY
					if (journalsummary = 1 and child.Summary != '') then

						-- ADD COLON BEFORE APPENDING TEXT TO THIS JOURNAL ENTRY
						if (thisjournalentry != '') then
							set thisjournalentry = thisjournalentry + ': ';
						end if;

						set thisjournalentry = thisjournalentry + child.Summary;
					end if;

					-- CHECK WHETHER OR NOT TO INCLUDE THE AlertKey IN THIS JOURNAL ENTRY
					if (journalalertkey = 1 and child.AlertKey != '') then

						-- ADD COLON BEFORE APPENDING TEXT TO THIS JOURNAL ENTRY
						if (thisjournalentry != '') then
							set thisjournalentry = thisjournalentry + ': ';
						end if;

						set thisjournalentry = thisjournalentry + child.AlertKey;
					end if;

					-- CHECK WHETHER OR NOT TO INCLUDE THE CustomText IN THIS JOURNAL ENTRY
					if (journalcustomtext = 1 and child.CustomText != '') then

						-- ADD COLON BEFORE APPENDING TEXT TO THIS JOURNAL ENTRY
						if (thisjournalentry != '') then
							set thisjournalentry = thisjournalentry + ': ';
						end if;

						set thisjournalentry = thisjournalentry + child.CustomText;
					end if;
				end if;

				-- ADD CURRENT EVENT TO ScopeIDParent JOURNAL ENTRY IF:
				-- - IF SENDING CHILD EVENT DATA AS JOURNALS TO ScopeIDParent IS ENABLED
				-- - IF THE MAXIMUM NUMBER OF EVENTS FOR THIS ScopeIDParent HAS NOT BEEN EXCEEDED
				-- - THE CURRENT JOURNAL ENTRY IS NOT NULL
				-- - THE CURRENT EVENT IS NOT A SiteNameParent EVENT
				-- - THE CURRENT EVENT IS NOT A ScopeIDParent EVENT
				-- - THE CURRENT EVENT IS NOT AN ASMParent EVENT
				if (journaltoscopeidparent = 1 and
					(scopeid.Poll + journalcounter) < maxscopeidjournals and
					thisjournalentry != '' and
					child.AlertGroup != 'SiteNameParent' and
					child.AlertGroup != 'ScopeIDParent' and
					child.AlertGroup != 'ASMParent') then

					-- APPEND A NEW LINE CHARACTER IF THERE IS ALREADY TEXT TO ADD
					if (scopeidparentjournalentry != '') then
						set scopeidparentjournalentry = scopeidparentjournalentry + '\n';
					end if;

					-- ADD THE CURRENT JOURNAL ENTRY TO THE EXISTING
					set scopeidparentjournalentry = scopeidparentjournalentry + thisjournalentry;

					-- INCREMENT THE NUMBER OF JOURNALS BEING SENT
					set journalcounter = journalcounter + 1;

					-- MARK CHILD EVENT AS HAVING BEEN JOURNALLED
					set child.JournalSent = 1;
 				end if;

				-- ADD A JOURNAL ENTRY TO THE CURRENT ScopeIDParent NOW IF MAX EVENTS PER ENTRY IS REACHED
				if (journalcounter = journalmaxeventsperentry) then

					EXECUTE jinsert(scopeid.Serial, 1000000 + journaloffset, getdate, 'CHILD EVENTS:\n' + scopeidparentjournalentry);

					-- UPDATE THE NUMBER OF CHILD EVENTS APPENDED TO THIS PARENT EVENT
					set scopeid.Poll = scopeid.Poll + journalcounter;

					-- APPEND NOTE THAT NO FURTHER JOURNALS WILL BE APPENDED TO THIS ScopeIDParent
					-- DUE TO THE MAXIMUM NUMBER HAVING BEEN REACHED
					if (scopeid.Poll = maxscopeidjournals) then

						EXECUTE jinsert(scopeid.Serial, 1000000 + journaloffset, getdate + 1,
							'*** MAXIMUM OF ' + to_char(maxscopeidjournals) +
							' EVENTS HAVE BEEN JOURNALED TO THIS PARENT');
					end if;

					-- INCREMENT THE JOURNAL UID OFFSET
					set journaloffset = journaloffset + 1;

					-- RESET SOME VARIABLES
					set scopeidparentjournalentry = '';
					set journalcounter = 0;
				end if;
			end if;
		end;

		-- UPDATE THE NUMBER OF ACTIVE CHILD EVENTS IN THE GRADE FIELD, IF NOT ZERO
		if ((alarmcounter2 != 0 or alarmcounter3 != 0) and scopeid.Grade != alarmcounter2 + alarmcounter3) then

			set scopeid.Grade = alarmcounter2 + alarmcounter3;

		-- ELSE UPDATE THE GRADE FIELD TO ZERO, IF IT IS NOT ALREADY
		elseif (alarmcounter2 = 0 and alarmcounter3 = 0 and scopeid.Grade != 0) then

			set scopeid.Grade = 0;
		end if;

		-- BEGIN THE CONSTRUCTION OF THE ScopeIDParent Summary FIELD

		-- SPECIFY WHETHER TO INCLUDE THE ScopeIDParent PREFIX
		if (usescopeidprefix = 1) then

			set summary = scopeidprefix;
		end if;

		-- SPECIFY WHETHER TO INCLUDE THE NUMBER OF ACTIVE ALARMS AT THE BEGINNING OF THE SUMMARY
		if (scopeidsummaryactivefirst = 1) then

			-- APPEND A SPACE CHARACTER IF summary NOT BLANK
			if (summary != '') then
				set summary = summary + ': ';
			end if;

			-- APPEND TOTAL NUMBER OF UNDERLYING REAL EVENTS
			if ((alarmcounter2 + alarmcounter3) = 1) then
				set summary = summary + '(' +
					to_char(alarmcounter2 + alarmcounter3) + ' active alarm)';
			else
				set summary = summary + '(' +
					to_char(alarmcounter2 + alarmcounter3) + ' active alarms)';
			end if;
		end if;

		-- SPECIFY WHETHER TO INCLUDE THE ScopeID LABEL
		if (usescopeidlabel = 1) then

			-- APPEND A COLON IF summary NOT BLANK
			if (summary != '') then
				set summary = summary + ': ';
			end if;

			set summary = summary + scopeid.ScopeID;
		end if;

		-- UPDATE PRIORITY CHILD DATA FOR THIS PARENT EVENT
		EXECUTE correlation_update_priority_child (scopeid.Identifier,
			propagatetexttoscopeidparentcause, highcauseweight, highcausetext,
			propagatetexttoscopeidparentimpact, highimpactweight, highimpacttext,
			propagatetexttoscopeidparentfirst, lowfirstoccurrence,
			propagatetexttoscopeidparentlast, highlastoccurrence,
			customtext);

		-- APPEND IMPACT CAUSE TEXT TO SUMMARY IF CONFIGURED
		if (usescopeidimpactcause = 1) then

			EXECUTE correlation_construct_impactcause (scopeid.Identifier, summary);
		end if;

		-- FETCH AND SET THE PRIORITY CHILD CauseWeight AND ImpactWeight VALUES
		if (	scopeid.Severity != 0 and
			(propagatetexttoscopeidparentcause = 1 or
			 propagatetexttoscopeidparentimpact = 1 or
			 propagatetexttoscopeidparentfirst = 1 or
			 propagatetexttoscopeidparentlast = 1 or
			 usescopeidimpactcause = 1)) then

			for each row parentrec in master.correlation_priority_children where parentrec.Identifier = scopeid.Identifier
			begin

				-- PROPAGATE PRIORITY CauseWeight TO PARENT IF IT HAS CHANGED
				if (scopeid.CauseWeight != parentrec.HighCauseWeight) then

					set scopeid.CauseWeight = parentrec.HighCauseWeight;
				end if;

				-- PROPAGATE PRIORITY ImpactWeight TO PARENT IF IT HAS CHANGED
				if (scopeid.ImpactWeight != parentrec.HighImpactWeight) then

					set scopeid.ImpactWeight = parentrec.HighImpactWeight;
				end if;
			end;
		end if;

		-- FETCH AND SET THE CURRENT PRIORITY CustomText IF ANY OF THE PROPAGATION OPTIONS ARE SELECTED
		if (	scopeid.Severity != 0 and
			(propagatetexttoscopeidparentcause = 1 or
			 propagatetexttoscopeidparentimpact = 1 or
			 propagatetexttoscopeidparentfirst = 1 or
			 propagatetexttoscopeidparentlast = 1)) then

			for each row parentrec in master.correlation_priority_children where parentrec.Identifier = scopeid.Identifier
			begin

				-- PROPAGATE PRIORITY CustomText TO PARENT IF IT HAS CHANGED
				if (scopeid.CustomText != parentrec.CustomText) then

					set scopeid.CustomText = parentrec.CustomText;
				end if;
			end;

			-- SPECIFY WHETHER TO INCLUDE THE CUSTOM SUMMARY TEXT CustomText
			if (usescopeidcustomtext = 1) then

				-- APPEND A COLON IF summary NOT BLANK
				if (summary != '') then
					set summary = summary + ': ';
				end if;

				-- APPEND CustomText
				set summary = summary + scopeid.CustomText;
			end if;
		end if;

		-- SPECIFY WHETHER TO INCLUDE THE NUMBER OF SITES AFFECTED
		-- DO NOT INCLUDE IF NO SITES ARE AFFECTED
		if (usescopeidsitesaffected = 1 and alarmcounter != 0) then

			-- APPEND A COLON IF summary NOT BLANK
			if (summary != '') then
				set summary = summary + ': ';
			end if;

			-- APPEND NUMBER OF SITES AFFECTED
			set summary = summary + to_char(alarmcounter) + ' ' + scopeidsitesaffectedlabel;

			if (alarmcounter = 1) then
				set summary = summary + ' affected';
			else
				set summary = summary + 's affected';
			end if;
		end if;

		-- SPECIFY WHETHER TO INCLUDE THE NUMBER OF ACTIVE ALARMS AT THE END OF THE SUMMARY
		if (usescopeidnumactivealarms = 1) then

			-- APPEND A SPACE CHARACTER IF summary NOT BLANK
			if (summary != '') then
				set summary = summary + ' ';
			end if;

			-- APPEND TOTAL NUMBER OF UNDERLYING REAL EVENTS
			if ((alarmcounter2 + alarmcounter3) = 1) then
				set summary = summary + '(' +
					to_char(alarmcounter2 + alarmcounter3) + ' active alarm)';
			else
				set summary = summary + '(' +
					to_char(alarmcounter2 + alarmcounter3) + ' active alarms)';
			end if;
		end if;

		-- UPDATE Synthetic Event - Parent EVENT IF NOT CLEAR - OR IF ANY OF THE COUNTERS ARE NOT ZERO
		if (scopeid.AlertGroup in ('Synthetic Event - Parent', 'ASMParent', 'CustomParent') and
			(scopeid.Severity != 0 or alarmcounter != 0 or alarmcounter2 != 0 or alarmcounter3 != 0)) then

			-- SET THE Severity TO INDETERMINITE IF UNDERLYING EVENTS ARE CLEARED
			-- THIS WILL EVENTUALLY BE CLEARED WHEN TIME WINDOW CLOSES
			if (highseverity = 0 and scopeid.Severity != 1) then

				set scopeid.Severity = 1;

			-- ELSE SET THE Severity TO THAT OF THE HIGHEST UNDERLYING EVENT
			elseif (scopeid.Severity != highseverity) then

				set scopeid.Severity = highseverity;
			end if;
		end if;

		-- UPDATE ScopeIDParent EVENT IF NOT CLEAR
		-- OR IF THE SUMMARY HAS CHANGED
		if (scopeid.AlertGroup = 'ScopeIDParent' and (scopeid.Severity != 0 or scopeid.Summary != summary)) then

			-- UPDATE Summary FIELD ONLY IF IT HAS CHANGED
			if (scopeid.Summary != summary) then

				set scopeid.Summary = summary;
			end if;

			-- SET THE Severity TO INDETERMINITE IF UNDERLYING EVENTS ARE CLEARED
			-- THIS WILL EVENTUALLY BE CLEARED WHEN TIME WINDOW CLOSES
			if (highseverity = 0 and scopeid.Severity != 1) then

				set scopeid.Severity = 1;

			-- ELSE SET THE Severity TO THAT OF THE HIGHEST UNDERLYING EVENT
			elseif (highseverity != 0 and scopeid.Severity != highseverity) then

				set scopeid.Severity = highseverity;
			end if;

			-- UPDATE FirstOccurrence BASED ON UNDERLYING EVENTS, IF NOT NULL
			if (lowfirstoccurrence != 0 and scopeid.FirstOccurrence != lowfirstoccurrence) then

				set scopeid.FirstOccurrence = lowfirstoccurrence;
			end if;

			-- UPDATE LastOccurrence BASED ON UNDERLYING EVENTS, IF NOT NULL
			if (highlastoccurrence != 0 and scopeid.LastOccurrence != highlastoccurrence) then

				set scopeid.LastOccurrence = highlastoccurrence;
			end if;
		end if;

		-- ADD A JOURNAL ENTRY TO THE CURRENT ScopeIDParent IF APPLICABLE
		if (journaltoscopeidparent = 1 and scopeidparentjournalentry != '') then

			EXECUTE jinsert(scopeid.Serial, 1000000 + journaloffset, getdate, 'CHILD EVENTS:\n' + scopeidparentjournalentry);

			-- UPDATE THE NUMBER OF CHILD EVENTS APPENDED TO THIS PARENT EVENT
			set scopeid.Poll = scopeid.Poll + journalcounter;

			-- APPEND NOTE THAT NO FURTHER JOURNALS WILL BE APPENDED TO THIS ScopeIDParent
			-- DUE TO THE MAXIMUM NUMBER HAVING BEEN REACHED
			if (scopeid.Poll = maxscopeidjournals) then

				EXECUTE jinsert(scopeid.Serial, 1000000 + journaloffset, getdate + 1,
					'*** MAXIMUM OF ' + to_char(maxscopeidjournals) +
					' EVENTS HAVE BEEN JOURNALED TO THIS PARENT');
			end if;
		end if;
	end;
end;
go

------------------------------------------------------------------------------
-- CREATE A PROCEDURE TO PROCESS SuperParent EVENTS
------------------------------------------------------------------------------

CREATE OR REPLACE PROCEDURE correlation_process_superparents ( )
declare

	journaltositenameparent INTEGER;	-- USED TO STORE PROPERTY FOR WHETHER OR NOT TO JOURNAL TO SiteNameParent EVENTS

	journaltoscopeidparent INTEGER;		-- USED TO STORE PROPERTY FOR WHETHER OR NOT TO JOURNAL TO ScopeIDParent EVENTS
	scopeidserial INTEGER;			-- USED TO STORE THE Serial OF THE CURRENT ScopeIDParent EVENT
	scopeidparentidentifier CHAR(255);	-- USED TO STORE THE ParentIdentifier OF THE CURRENT ScopeIDParent EVENT

	journaltosuperparent INTEGER;		-- USED TO STORE PROPERTY FOR WHETHER OR NOT TO JOURNAL TO SuperParent EVENTS
	superparentserial INTEGER;		-- USED TO STORE THE Serial OF THE CURRENT SuperParent EVENT
	superparentidentifier CHAR(255);		-- USED TO STORE THE ParentIdentifier OF THE CURRENT SuperParent EVENT
	superparentjournalcount INTEGER;	-- USED TO STORE THE NUMBER OF JOURNALS WRITTEN TO THE CURRENT SuperParent EVENT
	superparentjournalcounter INTEGER;	-- USED TO STORE HOW MANY EVENTS HAVE BEEN WRITTEN TO THE CURRENT superparentjournalentry
	maxsuperparentjournals INTEGER;		-- USED TO STORE THE MAXIMUM NUMBER OF JOURNALS ALLOWED IN THE CURRENT SuperParent EVENT

	journalmaxeventsperentry INTEGER;	-- USED TO STORE THE MAXIMUM NUMBER OF EVENTS THAT CAN BE WRITTEN IN EACH JOURNAL ENTRY

	journalservernameserverserial INTEGER;	-- USED TO STORE PROPERTY FOR WHETHER OR NOT TO WRITE AN EVENT ServerName AND ServerSerial TO JOURNAL
	journalnode INTEGER;			-- USED TO STORE PROPERTY FOR WHETHER OR NOT TO WRITE AN EVENT Node TO JOURNAL
	journalsummary INTEGER;			-- USED TO STORE PROPERTY FOR WHETHER OR NOT TO WRITE AN EVENT Summary TO JOURNAL
	journalalertkey INTEGER;		-- USED TO STORE PROPERTY FOR WHETHER OR NOT TO WRITE AN EVENT AlertKey TO JOURNAL
	journalcustomtext INTEGER;		-- USED TO STORE PROPERTY FOR WHETHER OR NOT TO WRITE AN EVENT CustomText TO JOURNAL

	thisjournalentry CHAR(4080);		-- USED TO CONSTRUCT THE ENTRY FOR THE CURRENT EVENT
	sitenameparentjournalentry CHAR(4080);	-- USED TO STORE THE CUMULATIVE JOURNAL ENTRY FOR A SiteNameParent EVENT
	sitenamejournalcounter INTEGER;		-- USED TO STORE HOW MANY EVENTS HAVE BEEN WRITTEN TO THE CURRENT sitenameparentjournalentry
	scopeidparentjournalentry CHAR(4080);	-- USED TO STORE THE CUMULATIVE JOURNAL ENTRY FOR A ScopeIDParent EVENT
	scopeidjournalcounter INTEGER;		-- USED TO STORE HOW MANY EVENTS HAVE BEEN WRITTEN TO THE CURRENT scopeidparentjournalentry
	sitenamejournaloffset INTEGER;		-- USED TO STORE THE CURRENT UID OFFSET TO USE WHEN WRITING SiteNameParent JOURNALS
	scopeidjournaloffset INTEGER;		-- USED TO STORE THE CURRENT UID OFFSET TO USE WHEN WRITING ScopeIDParent JOURNALS
begin
--
-- Procedure updates SuperParent like those created by NOI analytics to add journals of underlying child events.
-- This procedures uses properties stored in master.properties to drive its behaviour.
--
-- Called by: correlation_process_existing_parents
--
-- Usage:  EXECUTE correlation_process_superparents ( );
--

	-- INITIALISE LOCAL VARIABLES
	set journaltositenameparent = 0;

	set journaltoscopeidparent = 0;
	set scopeidserial = 0;
	set scopeidparentidentifier = '';

	set journaltosuperparent = 0;
	set superparentserial = 0;
	set superparentidentifier = '';
	set superparentjournalcount = 0;
	set superparentjournalcounter = 0;
	set maxsuperparentjournals = 100;

	set journalmaxeventsperentry = 20;

	set journalservernameserverserial = 1;
	set journalnode = 1;
	set journalsummary = 1;
	set journalalertkey = 1;
	set journalcustomtext = 1;

	set maxsuperparentjournals = 100;
	set superparentserial = 0;

	set thisjournalentry = '';
	set sitenameparentjournalentry = '';
	set scopeidparentjournalentry = '';
	set sitenamejournalcounter = 0;
	set scopeidjournalcounter = 0;
	set sitenamejournaloffset = 0;
	set scopeidjournaloffset = 0;

	-- LOAD UP VARIABLES BASED ON PROPERTIES
	for each row property in master.properties where property.Name in (
		'SEGJournalToSiteNameParent',
		'SEGJournalToScopeIDParent',
		'SEGJournalToSuperParent',
		'SEGJournalMaxEventsPerEntry',
		'SEGJournalServerNameServerSerial',
		'SEGJournalNode',
		'SEGJournalSummary',
		'SEGJournalAlertKey',
		'SEGJournalCustomText',
		'SEGMaxSiteNameJournals',
		'SEGMaxScopeIDJournals',
		'SEGMaxSuperParentJournals')
	begin


		-- SPECIFIES WHETHER OR NOT TO JOURNAL CHILD EVENTS TO THE SiteNameParent EVENT
		if (property.Name = 'SEGJournalToSiteNameParent') then

			set journaltositenameparent = property.IntValue;

		-- SPECIFIES WHETHER OR NOT TO JOURNAL CHILD EVENTS TO THE ScopeIDParent EVENT
		elseif (property.Name = 'SEGJournalToScopeIDParent') then

			set journaltoscopeidparent = property.IntValue;

		-- SPECIFIES WHETHER OR NOT TO JOURNAL CHILD EVENTS TO THE SuperParent EVENT
		elseif (property.Name = 'SEGJournalToSuperParent') then

			set journaltosuperparent = property.IntValue;

		-- SPECIFIES HOW MANY EVENTS TO ROLL UP INTO A SINGLE JOURNAL ENTRY
		elseif (property.Name = 'SEGJournalMaxEventsPerEntry') then

			set journalmaxeventsperentry = property.IntValue;

		-- SPECIFIES WHETHER OR NOT TO SEND THE ServerName:ServerSerial FIELDS TO THE PARENT EVENT JOURNAL
		elseif (property.Name = 'SEGJournalServerNameServerSerial') then

			set journalservernameserverserial = property.IntValue;

		-- SPECIFIES WHETHER OR NOT TO SEND THE Node FIELD TO THE PARENT EVENT JOURNAL
		elseif (property.Name = 'SEGJournalNode') then

			set journalnode = property.IntValue;

		-- SPECIFIES WHETHER OR NOT TO SEND THE Summary FIELD TO THE PARENT EVENT JOURNAL
		elseif (property.Name = 'SEGJournalSummary') then

			set journalsummary = property.IntValue;

		-- SPECIFIES WHETHER OR NOT TO SEND THE AlertKey FIELD TO THE PARENT EVENT JOURNAL
		elseif (property.Name = 'SEGJournalAlertKey') then

			set journalalertkey = property.IntValue;

		-- SPECIFIES WHETHER OR NOT TO SEND THE CustomText FIELD TO THE PARENT EVENT JOURNAL
		elseif (property.Name = 'SEGJournalCustomText') then

			set journalcustomtext = property.IntValue;

		-- SPECIFIES MAXIMUM NUMBER OF EVENTS TO SEND TO THE SUPER PARENT EVENT JOURNAL
		elseif (property.Name = 'SEGMaxSuperParentJournals') then

			set maxsuperparentjournals = property.IntValue;
		end if;
	end;

	-- ONLY WRITE JOURNALS TO SUPER PARENT IF SEGJournalToScopeIDParent = 1
	if (journaltosuperparent = 1) then

		-- FOR EACH ScopeIDParent
		for each row scopeid in alerts.status where scopeid.AlertGroup in ('ScopeIDParent', 'Synthetic Event - Parent', 'ASMParent', 'CustomParent')
		begin

			-- RESET VARIABLE
			set superparentjournalcounter = 0;

			-- ONLY PROCESS ScopeIDParent EVENTS THAT ARE THEMSELVES CHILDREN EVENTS
			if (scopeid.ParentIdentifier != '') then

				-- LOCATE SUPER PARENT AND STORE Serial VALUE
				for each row superparent in alerts.status where superparent.Identifier = scopeid.ParentIdentifier
				begin

					set superparentserial = superparent.Serial;
					set superparentidentifier = superparent.Identifier;
					set superparentjournalcount = superparent.Poll;
				end;

				-- INITIALISE VARIABLES
				set scopeidjournalcounter = 0;
				set scopeidparentjournalentry = '';

				-- ITERATE OVER CHILD EVENTS
				for each row child in alerts.status where child.ParentIdentifier = scopeid.Identifier
				begin

					-- DEAL WITH CASE WHERE CHILD IS A SiteNameParent EVENT
					if (child.AlertGroup = 'SiteNameParent') then

						-- INITIALISE VARIABLES
						set sitenamejournalcounter = 0;
						set sitenameparentjournalentry = '';

						-- ITERATE OVER GRANDCHILD EVENTS
						for each row grandchild in alerts.status where grandchild.ParentIdentifier = child.Identifier
						begin

							-- ADD GRANDCHILD TO JOURNAL ENTRY IF FLAGGED APPROPRIATELY - EITHER:
							-- - JOURNALLING TO SiteNameParent IS ENABLED AND IT HAS HAPPENED ALREADY
							-- - JOURNALLING TO SiteNameParent IS DISABLED AND HAS NOT BEEN JOURNALLED YET
							if (	(journaltositenameparent = 1 and grandchild.JournalSent = 1) OR
								(journaltositenameparent = 0 and grandchild.JournalSent = 0)) then

								-- CLEAR thisjournalentry CONTENTS AND counter VARIABLES
								set thisjournalentry = '';

								-- CONSTRUCT A JOURNAL ENTRY FOR CURRENT EVENT IF THE MAXIMUM NUMBER
								-- OF EVENTS PER JOURNAL ENTRY HAS NOT BEEN EXCEEDED
								if (sitenamejournalcounter < journalmaxeventsperentry) then

									-- CHECK WHETHER OR NOT TO INCLUDE THE ServerName:ServerSerial IN THIS JOURNAL ENTRY
									if (journalservernameserverserial = 1) then

										set thisjournalentry = thisjournalentry + grandchild.ServerName +
											':' + to_char(grandchild.ServerSerial);
									end if;

									-- CHECK WHETHER OR NOT TO INCLUDE THE Node IN THIS JOURNAL ENTRY
									if (journalnode = 1 and grandchild.Node != '') then

										-- ADD COLON BEFORE APPENDING TEXT TO THIS JOURNAL ENTRY
										if (thisjournalentry != '') then
											set thisjournalentry = thisjournalentry + ': ';
										end if;

										set thisjournalentry = thisjournalentry + grandchild.Node;
									end if;

									-- CHECK WHETHER OR NOT TO INCLUDE THE Summary IN THIS JOURNAL ENTRY
									if (journalsummary = 1 and grandchild.Summary != '') then

										-- ADD COLON BEFORE APPENDING TEXT TO THIS JOURNAL ENTRY
										if (thisjournalentry != '') then
											set thisjournalentry = thisjournalentry + ': ';
										end if;

										set thisjournalentry = thisjournalentry + grandchild.Summary;
									end if;

									-- CHECK WHETHER OR NOT TO INCLUDE THE AlertKey IN THIS JOURNAL ENTRY
									if (journalalertkey = 1 and grandchild.AlertKey != '') then

										-- ADD COLON BEFORE APPENDING TEXT TO THIS JOURNAL ENTRY
										if (thisjournalentry != '') then
											set thisjournalentry = thisjournalentry + ': ';
										end if;

										set thisjournalentry = thisjournalentry + grandchild.AlertKey;
									end if;

									-- CHECK WHETHER OR NOT TO INCLUDE THE CustomText IN THIS JOURNAL ENTRY
									if (journalcustomtext = 1 and grandchild.CustomText != '') then

										-- ADD COLON BEFORE APPENDING TEXT TO THIS JOURNAL ENTRY
										if (thisjournalentry != '') then
											set thisjournalentry = thisjournalentry + ': ';
										end if;

										set thisjournalentry = thisjournalentry + grandchild.CustomText;
									end if;
								end if;

								-- ADD CURRENT EVENT thisjournalentry TO sitenameparentjournalentry IF:
								-- - IF THE MAXIMUM NUMBER OF EVENTS PER JOURNAL ENTRY HAS NOT BEEN EXCEEDED
								-- - IF THE MAXIMUM NUMBER OF EVENTS FOR THIS SuperParent HAS NOT BEEN EXCEEDED
								-- - THE CURRENT JOURNAL ENTRY IS NOT NULL
								if (	sitenamejournalcounter < journalmaxeventsperentry and
									(superparentjournalcount + superparentjournalcounter + sitenamejournalcounter) < maxsuperparentjournals and
									thisjournalentry != '') then

									-- APPEND A NEW LINE CHARACTER IF THERE IS ALREADY TEXT TO ADD
									if (sitenameparentjournalentry != '') then
										set sitenameparentjournalentry = sitenameparentjournalentry + '\n';
									end if;

									-- ADD THE CURRENT JOURNAL ENTRY TO THE EXISTING
									set sitenameparentjournalentry = sitenameparentjournalentry + thisjournalentry;

									-- INCREMENT sitenamejournalcounter
									set sitenamejournalcounter = sitenamejournalcounter + 1;

									-- MARK CHILD EVENT AS HAVING BEEN JOURNALLED TO SUPER PARENT
									set grandchild.JournalSent = 2;
				 				end if;

								-- ADD A JOURNAL ENTRY TO THE CURRENT SuperParent NOW IF MAX EVENTS PER ENTRY IS REACHED, UPDATE COUNT, RESET VARS
								if (sitenamejournalcounter = journalmaxeventsperentry) then

									EXECUTE jinsert(superparentserial, 1100000 + sitenamejournaloffset, getdate,
										child.ScopeID + ': ' + child.SiteName + ': CHILD EVENTS:\n' + sitenameparentjournalentry);

									-- UPDATE THE NUMBER OF CHILD EVENTS APPENDED TO THIS PARENT EVENT
									set superparentjournalcounter = superparentjournalcounter + sitenamejournalcounter;

									-- APPEND NOTE THAT NO FURTHER JOURNALS WILL BE APPENDED TO THIS SuperParent
									-- DUE TO THE MAXIMUM NUMBER HAVING BEEN REACHED
									if (superparentjournalcount + superparentjournalcounter = maxsuperparentjournals) then

										EXECUTE jinsert(superparentserial, 1100000 + sitenamejournaloffset, getdate + 1,
											'*** MAXIMUM OF ' + to_char(maxsuperparentjournals) +
											' EVENTS HAVE BEEN JOURNALED TO THIS PARENT');
									end if;

									-- INCREMENT THE JOURNAL UID OFFSET
									set sitenamejournaloffset = sitenamejournaloffset + 1;

									-- RESET SOME VARIABLES
									set sitenameparentjournalentry = '';
									set sitenamejournalcounter = 0;
								end if;
							end if;

						end; -- ITERATE OVER GRANDCHILD EVENTS

						-- WRITE sitenamejournalentry TO SUPER PARENT IF NOT NULL
						if (sitenameparentjournalentry != '') then

							EXECUTE jinsert(superparentserial, 1100000 + sitenamejournaloffset, getdate,
								child.ScopeID + ': ' + child.SiteName + ': CHILD EVENTS:\n' + sitenameparentjournalentry);

							-- UPDATE THE NUMBER OF CHILD EVENTS APPENDED TO THIS PARENT EVENT
							set superparentjournalcounter = superparentjournalcounter + sitenamejournalcounter;

							-- APPEND NOTE THAT NO FURTHER JOURNALS WILL BE APPENDED TO THIS SiteNameParent
							-- DUE TO THE MAXIMUM NUMBER HAVING BEEN REACHED
							if (superparentjournalcount + superparentjournalcounter = maxsuperparentjournals) then

								EXECUTE jinsert(superparentserial, 1100000 + sitenamejournaloffset, getdate + 1,
									'*** MAXIMUM OF ' + to_char(maxsuperparentjournals) +
									' EVENTS HAVE BEEN JOURNALED TO THIS PARENT');
							end if;

							-- INCREMENT THE JOURNAL UID OFFSET
							set sitenamejournaloffset = sitenamejournaloffset + 1;
						end if;

					-- ELSE ScopeID CHILD EVENT IS A REGULAR EVENT
					else

						-- ADD CHILD TO JOURNAL ENTRY IF FLAGGED APPROPRIATELY - EITHER:
						-- - JOURNALLING TO ScopeIDParent IS ENABLED AND IT HAS HAPPENED ALREADY
						-- - JOURNALLING TO ScopeIDParent IS DISABLED AND HAS NOT BEEN JOURNALLED YET
						if (	(journaltoscopeidparent = 1 and child.JournalSent = 1) OR
							(journaltoscopeidparent = 0 and child.JournalSent = 0)) then

							-- CLEAR thisjournalentry VARIABLE
							set thisjournalentry = '';

							-- CONSTRUCT A JOURNAL ENTRY FOR CURRENT EVENT IF THE MAXIMUM NUMBER
							-- OF EVENTS PER JOURNAL ENTRY HAS NOT BEEN EXCEEDED
							if (scopeidjournalcounter < journalmaxeventsperentry) then

								-- CHECK WHETHER OR NOT TO INCLUDE THE ServerName:ServerSerial IN THIS JOURNAL ENTRY
								if (journalservernameserverserial = 1) then

									set thisjournalentry = thisjournalentry + child.ServerName +
										':' + to_char(child.ServerSerial);
								end if;

								-- CHECK WHETHER OR NOT TO INCLUDE THE Node IN THIS JOURNAL ENTRY
								if (journalnode = 1 and child.Node != '') then

									-- ADD COLON BEFORE APPENDING TEXT TO THIS JOURNAL ENTRY
									if (thisjournalentry != '') then
										set thisjournalentry = thisjournalentry + ': ';
									end if;

									set thisjournalentry = thisjournalentry + child.Node;
								end if;

								-- CHECK WHETHER OR NOT TO INCLUDE THE Summary IN THIS JOURNAL ENTRY
								if (journalsummary = 1 and child.Summary != '') then

									-- ADD COLON BEFORE APPENDING TEXT TO THIS JOURNAL ENTRY
									if (thisjournalentry != '') then
										set thisjournalentry = thisjournalentry + ': ';
									end if;

									set thisjournalentry = thisjournalentry + child.Summary;
								end if;

								-- CHECK WHETHER OR NOT TO INCLUDE THE AlertKey IN THIS JOURNAL ENTRY
								if (journalalertkey = 1 and child.AlertKey != '') then

									-- ADD COLON BEFORE APPENDING TEXT TO THIS JOURNAL ENTRY
									if (thisjournalentry != '') then
										set thisjournalentry = thisjournalentry + ': ';
									end if;

									set thisjournalentry = thisjournalentry + child.AlertKey;
								end if;

								-- CHECK WHETHER OR NOT TO INCLUDE THE CustomText IN THIS JOURNAL ENTRY
								if (journalcustomtext = 1 and child.CustomText != '') then

									-- ADD COLON BEFORE APPENDING TEXT TO THIS JOURNAL ENTRY
									if (thisjournalentry != '') then
										set thisjournalentry = thisjournalentry + ': ';
									end if;

									set thisjournalentry = thisjournalentry + child.CustomText;
								end if;
							end if;

							-- ADD CURRENT EVENT thisjournalentry TO scopeidparentjournalentry IF:
							-- - IF THE MAXIMUM NUMBER OF EVENTS PER JOURNAL ENTRY HAS NOT BEEN EXCEEDED
							-- - IF THE MAXIMUM NUMBER OF EVENTS FOR THIS SuperParent HAS NOT BEEN EXCEEDED
							-- - THE CURRENT JOURNAL ENTRY IS NOT NULL
							if (	scopeidjournalcounter < journalmaxeventsperentry and
								(superparentjournalcount + superparentjournalcounter + scopeidjournalcounter) < maxsuperparentjournals and
								thisjournalentry != '') then

								-- APPEND A NEW LINE CHARACTER IF THERE IS ALREADY TEXT TO ADD
								if (scopeidparentjournalentry != '') then
									set scopeidparentjournalentry = scopeidparentjournalentry + '\n';
								end if;

								-- ADD THE CURRENT JOURNAL ENTRY TO THE EXISTING
								set scopeidparentjournalentry = scopeidparentjournalentry + thisjournalentry;

								-- INCREMENT sitenamejournalcounter
								set scopeidjournalcounter = scopeidjournalcounter + 1;

								-- MARK CHILD EVENT AS HAVING BEEN JOURNALLED TO SUPER PARENT
								set child.JournalSent = 2;
			 				end if;

							-- ADD A JOURNAL ENTRY TO THE CURRENT SuperParent NOW IF MAX EVENTS PER ENTRY IS REACHED, UPDATE COUNT, RESET VARS
							if (scopeidjournalcounter = journalmaxeventsperentry) then

								EXECUTE jinsert(superparentserial, 1000000 + scopeidjournaloffset, getdate,
									scopeid.ScopeID + ': CHILD EVENTS:\n' + scopeidparentjournalentry);

								-- UPDATE THE NUMBER OF CHILD EVENTS APPENDED TO THIS PARENT EVENT
								set superparentjournalcounter = superparentjournalcounter + scopeidjournalcounter;

								-- APPEND NOTE THAT NO FURTHER JOURNALS WILL BE APPENDED TO THIS SiteNameParent
								-- DUE TO THE MAXIMUM NUMBER HAVING BEEN REACHED
								if (superparentjournalcount + superparentjournalcounter = maxsuperparentjournals) then

									EXECUTE jinsert(superparentserial, 1000000 + scopeidjournaloffset, getdate + 1,
										'*** MAXIMUM OF ' + to_char(maxsuperparentjournals) +
										' EVENTS HAVE BEEN JOURNALED TO THIS PARENT');
								end if;

								-- INCREMENT THE JOURNAL UID OFFSET
								set scopeidjournaloffset = scopeidjournaloffset + 1;

								-- RESET SOME VARIABLES
								set scopeidparentjournalentry = '';
								set scopeidjournalcounter = 0;
							end if;
						end if;
					end if;

				end; -- ITERATE OVER CHILD EVENTS

				-- WRITE scopeidjournalentry TO SUPER PARENT IF NOT NULL
				if (scopeidparentjournalentry != '') then

					EXECUTE jinsert(superparentserial, 1000000 + scopeidjournaloffset, getdate,
						scopeid.ScopeID + ': CHILD EVENTS:\n' + scopeidparentjournalentry);

					-- UPDATE THE NUMBER OF CHILD EVENTS APPENDED TO THIS PARENT EVENT
					set superparentjournalcounter = superparentjournalcounter + scopeidjournalcounter;

					-- APPEND NOTE THAT NO FURTHER JOURNALS WILL BE APPENDED TO THIS SiteNameParent
					-- DUE TO THE MAXIMUM NUMBER HAVING BEEN REACHED
					if (superparentjournalcount + superparentjournalcounter = maxsuperparentjournals) then

						EXECUTE jinsert(superparentserial, 1000000 + scopeidjournaloffset, getdate + 1,
							'*** MAXIMUM OF ' + to_char(maxsuperparentjournals) +
							' EVENTS HAVE BEEN JOURNALED TO THIS PARENT');
					end if;

					-- INCREMENT THE JOURNAL UID OFFSET
					set scopeidjournaloffset = scopeidjournaloffset + 1;
				end if;
			end if;

			-- UPDATE TOTAL NUMBER OF JOURNALS STORED IN SUPER PARENT IF ANY HAVE BEEN ADDED
			if (superparentjournalcounter != 0) then

				update alerts.status set Poll = Poll + superparentjournalcounter where Identifier = superparentidentifier;
			end if;

		end; -- FOR EACH ScopeIDParent
	end if;
end;
go

------------------------------------------------------------------------------
-- CREATE A TRIGGER TO UPDATE EXISTING SYNTHETIC PARENT EVENTS
------------------------------------------------------------------------------

CREATE OR REPLACE TRIGGER correlation_process_existing_parents
GROUP correlation_triggers
PRIORITY 15
COMMENT 'Update any existing synthetic parent events that are present'
EVERY 17 SECONDS
WHEN get_prop_value('ActingPrimary') %= 'TRUE'
begin

-- STEP 1: DISMANTLE ANY SMALL GROUPS THAT HAVE EXPIRED AND DON'T HAVE MINIMUM NUMBER OF CHILDREN

	EXECUTE correlation_dismantle_small_groups;

-- STEP 2: CLEAR PARENT EVENTS THAT HAVE NO CHILDREN AND ARE PAST THEIR EXPIRY TIMES

	EXECUTE correlation_clear_expired_parents;

-- STEP 3: PROCESS SiteNameParent EVENTS

	EXECUTE correlation_process_sitenameparents;

-- STEP 4: PROCESS ScopeIDParent EVENTS

	EXECUTE correlation_process_scopeidparents;

-- STEP 5: PROPAGATE CHILD AND GRANDCHILD EVENTS TO SUPER PARENT JOURNAL

	EXECUTE correlation_process_superparents;

end;
go










