------------------------------------------------------------------------------
--
--	Licensed Materials - Property of IBM
--
--	5724O4800
--
--	(C) Copyright IBM Corp. 2016. All Rights Reserved
--
--	US Government Users Restricted Rights - Use, duplication
--	or disclosure restricted by GSA ADP Schedule Contract
--	with IBM Corp.
--
--	Netcool/OMNIbus 8.1.0 - July 2018  Build: 5.50.78
--
--	SCOPE BASED EVENT GROUPING
--
--	This SQL file adds the fields and triggers used by the related event grouping
--
------------------------------------------------------------------------------

------------------------------------------------------------------------------
--
-- UPDATE SCRIPT
--
-- THIS SQL SCRIPT UPDATES SCOPE-BASED EVENT GROUPING RELEASED IN 8.1 FP4 TO FP7
--
-- NOTE: THIS SHOULD ONLY BE APPLIED TO AN AGGREGATION OBJECTSERVER
--
------------------------------------------------------------------------------

------------------------------------------------------------------------------
-- CREATE NEW FIELDS IN AGGREGATION OBJECTSERVER
------------------------------------------------------------------------------

ALTER TABLE alerts.status ADD COLUMN CustomText VARCHAR(255);
go

ALTER TABLE alerts.status ADD COLUMN JournalSent INTEGER;
go

------------------------------------------------------------------------------
-- CREATE NEW INDEX ON alerts.status TABLE
-----------------------------------------------------------------------------

CREATE INDEX AlertGroupIdx ON alerts.status USING HASH (AlertGroup);
go

------------------------------------------------------------------------------
-- REMOVE DEFUNCT PROPERTIES

delete from master.properties where Name in (
	'QuietPeriod',
	'PropagateTTNumber',
	'PropagateAcknowledged');
go

------------------------------------------------------------------------------
-- SAVE CODE VERSION IN OBJECTSERVER

insert into master.properties (Name, CharValue) values (
'SEGVersion', 
'Netcool/OMNIbus 8.1.0 - July 2018  Build: 5.50.78');
go

------------------------------------------------------------------------------

-- CREATE PROPERTIES TO BE USED BY THE AUTOMATIONS

-- USED TO DEFINE A GLOBAL QUIETPERIOD TO USE WHEN NOT DEFINED IN THE EVENT
insert into master.properties (Name, IntValue) values ('SEGQuietPeriod', (15 * 60));
go

-- USED TO SPECIFY WHETHER OR NOT TO PROPAGATE THE TTNumber TO CHILDREN EVENTS
insert into master.properties (Name, IntValue) values ('SEGPropagateTTNumber', 1);
go

-- USED TO SPECIFY WHETHER OR NOT TO PROPAGATE THE ACKNOWLEDGED STATUS TO CHILDREN EVENTS
insert into master.properties (Name, IntValue) values ('SEGPropagateAcknowledged', 1);
go

-- USED TO SPECIFY WHETHER OR NOT TO PROPAGATE THE OwnerUID TO CHILDREN EVENTS
insert into master.properties (Name, IntValue) values ('SEGPropagateOwnerUID', 1);
go

-- USED TO SPECIFY WHETHER OR NOT TO PROPAGATE THE OwnerGID TO CHILDREN EVENTS
insert into master.properties (Name, IntValue) values ('SEGPropagateOwnerGID', 1);
go

-- IF ENABLED, WILL USE THE Node OF THE FIRST EVENT FOR THE GROUPING TO POPULATE THE Node FIELD
-- IN THE SYNTHETIC ScopeIDParent EVENT
insert into master.properties (Name, IntValue) values ('SEGUseNodeForScopeIDParent', 0);
go

-- IF ENABLED, WILL USE THE Node OF THE FIRST EVENT FOR THE GROUPING TO POPULATE THE Node FIELD
-- IN THE SYNTHETIC SiteNameParent EVENT
insert into master.properties (Name, IntValue) values ('SEGUseNodeForSiteNameParent', 0);
go

-- IF ENABLED, WILL NOT CREATE A SiteNameParent SYNTHETIC EVENT, IF THE SiteName FIELD OF
-- THE INCOMING EVENTS IS BLANK.  EVENTS THAT HAVE SiteName SET WILL STILL BE SUB-GROUPED
-- UNDER SiteNameParent EVENTS.  BOTH TYPES OF CHILD EVENT CAN EXIST UNDER THE SAME ScopeIDParent
-- AT THE SAME TIME
insert into master.properties (Name, IntValue) values ('SEGNoSiteNameParentIfSiteNameBlank', 1);
go

-- SPECIFIES WHETHER OR NOT TO APPEND THE ScopeID PREFIX TO THE ScopeIDParent SUMMARY FIELD
insert into master.properties (Name, IntValue) values ('SEGUseScopeIDPrefix', 1);
go

-- SPECIFIES WHAT THE ScopeID PREFIX SHOULD BE, IF INCLUDED IN THE ScopeIDParent SUMMARY FIELD
insert into master.properties (Name, CharValue) values ('SEGScopeIDPrefix', 'INCIDENT');
go

-- SPECIFIES WHETHER OR NOT TO APPEND THE ScopeID LABEL TO THE ScopeIDParent SUMMARY FIELD
insert into master.properties (Name, IntValue) values ('SEGUseScopeIDLabel', 1);
go

-- SPECIFIES WHETHER OR NOT TO APPEND THE IMPACT AND CAUSE TEXT TO THE ScopeIDParent SUMMARY FIELD
-- THIS WILL ONLY BE OF USE IF THERE ARE CHILD EVENTS DIRECTLY UNDER THE ScopeIDParent EVENT
-- IT WILL NOT LOOK AT CHILD EVENTS CONTAINED UNDER SiteNameParent SUB-GROUPS
insert into master.properties (Name, IntValue) values ('SEGUseScopeIDImpactCause', 0);
go

-- SPECIFIES WHETHER OR NOT TO APPEND THE CustomText TO THE ScopeIDParent SUMMARY FIELD
insert into master.properties (Name, IntValue) values ('SEGUseScopeIDCustomText', 0);
go

-- SPECIFIES WHETHER OR NOT TO POPULATE THE CustomText OF THE ScopeIDParent EVENT USING
-- THE CustomText OF THE UNDERLYING CHILD EVENT WITH THE HIGHEST CauseWeight VALUE
insert into master.properties (Name, IntValue) values ('SEGPropagateTextToScopeIDParentCause', 0);
go

-- SPECIFIES WHETHER OR NOT TO POPULATE THE CustomText OF THE ScopeIDParent EVENT USING
-- THE CustomText OF THE UNDERLYING CHILD EVENT WITH THE HIGHEST CauseWeight VALUE
insert into master.properties (Name, IntValue) values ('SEGPropagateTextToScopeIDParentImpact', 0);
go

-- SPECIFIES WHETHER OR NOT TO POPULATE THE CustomText OF THE ScopeIDParent EVENT USING
-- THE CustomText OF THE UNDERLYING CHILD EVENT WITH THE LOWEST FirstOccurrence VALUE
insert into master.properties (Name, IntValue) values ('SEGPropagateTextToScopeIDParentFirst', 0);
go

-- SPECIFIES WHETHER OR NOT TO POPULATE THE CustomText OF THE ScopeIDParent EVENT USING
-- THE CustomText OF THE UNDERLYING CHILD EVENT WITH THE HIGHEST LastOccurrence VALUE
insert into master.properties (Name, IntValue) values ('SEGPropagateTextToScopeIDParentLast', 0);
go

-- SPECIFIES WHETHER OR NOT TO APPEND THE NUMBER OF SITES AFFECTED TEXT TO THE ScopeIDParent SUMMARY FIELD
insert into master.properties (Name, IntValue) values ('SEGUseScopeIDSitesAffected', 1);
go

-- SPECIFIES WHETHER OR NOT TO APPEND THE NUMBER OF ACTIVE ALARMS TEXT TO THE ScopeIDParent SUMMARY FIELD
insert into master.properties (Name, IntValue) values ('SEGUseScopeIDNumActiveAlarms', 1);
go

-- SPECIFIES WHETHER OR NOT TO APPEND THE SiteName PREFIX TO THE SiteNameParent SUMMARY FIELD
insert into master.properties (Name, IntValue) values ('SEGUseSiteNamePrefix', 1);
go

-- SPECIFIES WHAT THE SiteName PREFIX SHOULD BE, IF INCLUDED IN THE SiteNameParent SUMMARY FIELD
insert into master.properties (Name, CharValue) values ('SEGSiteNamePrefix', 'SITE');
go

-- SPECIFIES WHETHER OR NOT TO APPEND THE SiteName LABEL TO THE SiteNameParent SUMMARY FIELD
insert into master.properties (Name, IntValue) values ('SEGUseSiteNameLabel', 1);
go

-- SPECIFIES WHETHER OR NOT TO APPEND THE IMPACT AND CAUSE TEXT TO THE SiteNameParent SUMMARY FIELD
insert into master.properties (Name, IntValue) values ('SEGUseSiteNameImpactCause', 1);
go

-- SPECIFIES WHETHER OR NOT TO APPEND THE CustomText TO THE SiteNameParent SUMMARY FIELD
insert into master.properties (Name, IntValue) values ('SEGUseSiteNameCustomText', 1);
go

-- SPECIFIES WHETHER OR NOT TO POPULATE THE CustomText OF THE SiteNameParent EVENT USING
-- THE CustomText OF THE UNDERLYING CHILD EVENT WITH THE HIGHEST CauseWeight VALUE
insert into master.properties (Name, IntValue) values ('SEGPropagateTextToSiteNameParentCause', 0);
go

-- SPECIFIES WHETHER OR NOT TO POPULATE THE CustomText OF THE SiteNameParent EVENT USING
-- THE CustomText OF THE UNDERLYING CHILD EVENT WITH THE HIGHEST CauseWeight VALUE
insert into master.properties (Name, IntValue) values ('SEGPropagateTextToSiteNameParentImpact', 0);
go

-- SPECIFIES WHETHER OR NOT TO POPULATE THE CustomText OF THE SiteNameParent EVENT USING
-- THE CustomText OF THE UNDERLYING CHILD EVENT WITH THE LOWEST FirstOccurrence VALUE
insert into master.properties (Name, IntValue) values ('SEGPropagateTextToSiteNameParentFirst', 0);
go

-- SPECIFIES WHETHER OR NOT TO POPULATE THE CustomText OF THE SiteNameParent EVENT USING
-- THE CustomText OF THE UNDERLYING CHILD EVENT WITH THE HIGHEST LastOccurrence VALUE
insert into master.properties (Name, IntValue) values ('SEGPropagateTextToSiteNameParentLast', 0);
go

-- SPECIFIES WHETHER OR NOT TO APPEND THE NUMBER OF ACTIVE ALARMS TEXT TO THE SiteNameParent SUMMARY FIELD
insert into master.properties (Name, IntValue) values ('SEGUseSiteNameNumActiveAlarms', 1);
go

-- SPECIFIES THE UID TO USE WHEN INSERTING JOURNALS INTO SYNTHETIC PARENT EVENTS - DEFAULT IS ROOT
insert into master.properties (Name, IntValue) values ('SEGJournalUID', 0);
go

-- SPECIFIES WHETHER OR NOT TO JOURNAL CHILD EVENTS TO THE SiteNameParent EVENT
insert into master.properties (Name, IntValue) values ('SEGJournalToSiteNameParent', 0);
go

-- SPECIFIES WHETHER OR NOT TO JOURNAL CHILD EVENTS TO THE ScopeIDParent EVENT
insert into master.properties (Name, IntValue) values ('SEGJournalToScopeIDParent', 0);
go

-- SPECIFIES HOW MANY EVENTS TO ROLL UP INTO A SINGLE JOURNAL ENTRY
insert into master.properties (Name, IntValue) values ('SEGJournalMaxEventsPerEntry', 20);
go

-- SPECIFIES WHETHER OR NOT TO SEND THE ServerName:ServerSerial FIELDS TO THE PARENT EVENT JOURNAL
insert into master.properties (Name, IntValue) values ('SEGJournalServerNameServerSerial', 1);
go

-- SPECIFIES WHETHER OR NOT TO SEND THE Node FIELD TO THE PARENT EVENT JOURNAL
insert into master.properties (Name, IntValue) values ('SEGJournalNode', 1);
go

-- SPECIFIES WHETHER OR NOT TO SEND THE Summary FIELD TO THE PARENT EVENT JOURNAL
insert into master.properties (Name, IntValue) values ('SEGJournalSummary', 1);
go

-- SPECIFIES WHETHER OR NOT TO SEND THE AlertKey FIELD TO THE PARENT EVENT JOURNAL
insert into master.properties (Name, IntValue) values ('SEGJournalAlertKey', 0);
go

-- SPECIFIES WHETHER OR NOT TO SEND THE CustomText FIELD TO THE PARENT EVENT JOURNAL
insert into master.properties (Name, IntValue) values ('SEGJournalCustomText', 0);
go

-- SPECIFIES MAXIMUM NUMBER OF EVENTS TO SEND TO THE SiteNameParent EVENT JOURNAL
insert into master.properties (Name, IntValue) values ('SEGMaxSiteNameJournals', 10);
go

-- SPECIFIES MAXIMUM NUMBER OF EVENTS TO SEND TO THE ScopeIDParent EVENT JOURNAL
insert into master.properties (Name, IntValue) values ('SEGMaxScopeIDJournals', 50);
go

-- SPECIFIES THE MINIMUM NUMBER OF CHILDREN A ScopeIDParent MUST HAVE TO LIVE BEYOND ITS EXPIRETIME
insert into master.properties (Name, IntValue) values('SEGMinimumChildren', 0);
go

-- SPECIFIES THE MINIMUM TIME TO LIVE OF ScopeIDParent EVENTS - USED BY THE MINIMUM CHILDREN AUTOMATION
insert into master.properties (Name, IntValue) values('SEGMinimumScopeIDTTL', 120);
go

-- SPECIFIES THAT INHERITED TICKET NUMBER SHOULD BE CLEARED FROM CHILDREN IF DISMANTLING
insert into master.properties (Name, IntValue) values('SEGClearTTNumber', 0);
go

------------------------------------------------------------------------------
-- CREATE INSERT TRIGGER ON alerts.status TO CREATE PARENT EVENTS AND LINK CHILDREN
------------------------------------------------------------------------------

CREATE OR REPLACE TRIGGER correlation_new_row
GROUP correlation_triggers
PRIORITY 15
COMMENT 'Checks for the existence of parent events, updates or creates if necessary'
BEFORE INSERT ON alerts.status
FOR EACH ROW
WHEN get_prop_value('ActingPrimary') %= 'TRUE' and
	new.ScopeID != '' and
	new.ParentIdentifier = '' and
	new.AlertGroup not in ('SiteNameParent', 'ScopeIDParent', 'Synthetic Event - Parent')
declare

	quietperiod		INTEGER;
	usenodeforscopeidparent	INTEGER;
	usenodeforsitenameparent INTEGER;
	nodetouse		CHAR(64);
	nositenameparentifsitenameblank INTEGER;

	now			TIME;
	expiretime		TIME;

	scopeid			CHAR(255);

	scopeparentfound	INTEGER;
	newscopeparentneeded	INTEGER;
	scopeidentifier		CHAR(255);

	siteparentfound		INTEGER;
	scopeparentmoved	INTEGER;
	siteidentifier		CHAR(255);

begin

-- STEP 1: INITIALISE VARIABLES

	set quietperiod = 15 * 60;
	set usenodeforscopeidparent = 0;
	set usenodeforsitenameparent = 0;
	set nodetouse = '';
	set nositenameparentifsitenameblank = 0;

	set now = getdate();

	set scopeid = '';

	set scopeparentfound = 0;
	set newscopeparentneeded = 0;
	set scopeidentifier = '';

	set siteparentfound = 0;
	set scopeparentmoved = 0;
	set siteidentifier = '';

	-- SET UP VARIABLES BASED ON PROPERTIES
	for each row property in master.properties where property.Name in (
		'SEGQuietPeriod',
		'SEGUseNodeForScopeIDParent',
		'SEGUseNodeForSiteNameParent',
		'SEGNoSiteNameParentIfSiteNameBlank')
	begin
		-- SPECIFIES THE GLOBAL DEFAULT QUIET PERIOD TO APPLY TO GROUPING
		if (property.Name = 'SEGQuietPeriod') then

			set quietperiod = property.IntValue;

		-- SPECIFIES IF THE NODE VALUE OF THE FIRST EVENT SHOULD BE USED FOR THE ScopeIDParent EVENT
		elseif (property.Name = 'SEGUseNodeForScopeIDParent') then

			set usenodeforscopeidparent = property.IntValue;

		-- SPECIFIES IF THE NODE VALUE OF THE FIRST EVENT SHOULD BE USED FOR THE SiteNameParent EVENT
		elseif (property.Name = 'SEGUseNodeForSiteNameParent') then

			set usenodeforsitenameparent = property.IntValue;

		-- SPECIFIES WHETHER OR NOT A SiteNameParent SHOULD BE CREATED IF SiteName IS NULL
		elseif (property.Name = 'SEGNoSiteNameParentIfSiteNameBlank') then

			set nositenameparentifsitenameblank = property.IntValue;

		end if;
	end;

	-- OVERRIDE QUIET PERIOD IF DEFINED IN THE INCOMING EVENT
	if (new.QuietPeriod != 0) then

		set quietperiod = new.QuietPeriod;
	end if;

	-- USE VALUE STORED IN FirstOccurrence IF SET, ELSE USE CURRENT TIME
	if (new.FirstOccurrence = 0) then

		set now = getdate();
	else 

		set now = new.FirstOccurrence;
	end if;

	-- SET DEFAULT VALUE OF ExpireTime BASED ON FirstOccurrence
	set expiretime = now + quietperiod;

-- STEP 2: CHECK ScopeAlias MEMBERSHIP

	-- CHECK FOR ScopeAlias MEMBERSHIP
	for each row scopealiasmemberrow in master.correlation_scopealias_members where
		scopealiasmemberrow.ScopeID = new.ScopeID
	begin

		-- SET THE ScopeID TO USE FOR THE SYNTHETIC CONTAINMENT EVENT TO BE THE ALIAS
		set scopeid = scopealiasmemberrow.ScopeAlias;
	end;

	-- USE INCOMING ScopeID VALUE IF ScopeID IS NOT A MEMBER OF AN ALIAS GROUP
	if (scopeid = '') then

		set scopeid = new.ScopeID;
	end if;

-- STEP 3: SET UP ScopeIDParent EVENT

	-- LOOK FOR CURRENT ScopeIDParent
	for each row parent in master.correlation_scopeid where
		parent.ScopeID = scopeid
	begin

		-- MARK ScopeID PARENT AS FOUND
		set scopeparentfound = 1;

		-- CHECK IF THE GROUP ExpireTime HAS PASSED
		if (parent.ExpireTime < now) then

			-- IF SO, UPDATE PARENT ENTRY AND SET FLAG TO CREATE A NEW ScopeIDParent
			set parent.Identifier =	'ScopeIDParent:' + parent.ScopeID + ':' + to_char(now);
			set newscopeparentneeded = 1;
		end if;

		-- STORE EITHER EXISTING OR UPDATED ScopeIDParent IDENTIFIER
		set scopeidentifier = parent.Identifier;

		-- UPDATE LastOccurrence IN PARENT ENTRY IF INCOMING VALUE IS GREATER
		if (parent.LastOccurrence < now) then

			set parent.LastOccurrence = now;
		end if;

		-- UPDATE GROUP ExpireTime IF INCOMING ROW EXTENDS IT
		if (parent.ExpireTime < expiretime) then

			set parent.ExpireTime = expiretime;
		end if;
	end;

	-- CREATE PARENT ENTRY IN master.correlation_scopeid IF NONE FOUND
	if (scopeparentfound = 0) then

		-- SET ScopeIDParent IDENTIFIER TO A NEW VALUE
		set scopeidentifier = 'ScopeIDParent:' + scopeid + ':' + to_char(now);

		-- INSERT NEW ScopeIDParent
		insert into master.correlation_scopeid (
			ScopeID,
			LastOccurrence,
			Identifier,
			ExpireTime)
		values (
			scopeid,
			now,
			scopeidentifier,
			expiretime);
	end if;

	-- SET THE NODE FIELD VALUE TO USE BASED ON PROPERTY
	if (usenodeforscopeidparent = 1) then

		set nodetouse = new.Node;
	else

		set nodetouse = scopeid;
	end if;

	-- INSERT ScopeIDParent EVENT INTO alerts.status IF NOT PRESENT OR NEW ONE NEEDED
	if (scopeparentfound = 0 or newscopeparentneeded = 1) then

		-- INSERT SYNTHETIC ScopeIDParent EVENT
		insert into alerts.status (
			Identifier,
			Node,
			Class,
			Summary,
			AlertGroup,
			Severity,
			ScopeID,
			FirstOccurrence,
			LastOccurrence,
			Grade,
			OwnerUID,
			OwnerGID)
		values (
			scopeidentifier,
			nodetouse,
			99990,
			'INCIDENT: ' + scopeid + ': calculating summary details...',
			'ScopeIDParent',
			1,
			scopeid,
			now,
			now,
			1,
			65534,
			0);
	end if;

-- STEP 4: SET UP SiteNameParent EVENT IF OPTION TO DISABLE IS NOT SET

	-- DO NOT CREATE SiteNameParent EVENT IF OPTION TO DISABLE SiteNameParent
	-- IS SET AND SiteName IS BLANK.  INSTEAD LINK EVENT TO ScopeIDParent EVENT
	if (nositenameparentifsitenameblank = 1 and new.SiteName = '') then

		-- LINK CURRENT EVENT TO ScopeIDParent EVENT
		set new.ParentIdentifier = scopeidentifier;

	-- ELSE CREATE SiteNameParent EVENT AND LINK CURRENT EVENT TO THAT
	else

		-- IF EVENT HAS NO SiteName THEN SET IT TO A DEFAULT
		if (new.SiteName = '') then

			set new.SiteName = 'NO SITENAME';
		end if;

		-- LOOK FOR CURRENT SiteNameParent
		for each row parent in master.correlation_sitename where
			parent.SiteName = new.SiteName and
			parent.ScopeID = scopeid
		begin

			-- MARK SiteName PARENT AS FOUND
			set siteparentfound = 1;

			-- CHECK IF ScopeIDParent HAS MOVED
			for each row scopeidparentevent in alerts.status where
				scopeidparentevent.Identifier = parent.Identifier
			begin

				if (scopeidparentevent.ParentIdentifier != scopeidentifier) then

					set scopeparentmoved = 1;
				end if;
			end;

			-- UPDATE THE ENTRY IF WE ARE CREATING A NEW ScopeIDParent
			if (newscopeparentneeded = 1 or scopeparentmoved = 1) then

				set parent.Identifier = 'SiteNameParent:' +
					scopeid + ':' + new.SiteName + ':' + to_char(now);
			end if;

			-- STORE CURRENT SiteName PARENT IDENTIFIER
			set siteidentifier = parent.Identifier;
		end;

		-- CREATE PARENT ENTRY IN master.correlation_sitename IF NONE FOUND
		if (siteparentfound = 0) then

			-- STORE NEW SiteName PARENT IDENTIFIER
			set siteidentifier = 'SiteNameParent:' +
				scopeid + ':' + new.SiteName + ':' + to_char(now);

			-- CREATE THE NEW SiteParent EVENT
			insert into master.correlation_sitename (
				SiteName,
				ScopeID,
				Identifier)
			values (
				new.SiteName,
				scopeid,
				siteidentifier);
		end if;

		-- SET THE NODE FIELD VALUE TO USE BASED ON PROPERTY
		if (usenodeforsitenameparent = 1) then

			set nodetouse = new.Node;
		else

			set nodetouse = new.SiteName;
		end if;

		-- INSERT NEW SiteParent EVENT INTO alerts.status IF NEW OR UPDATED ScopeIDParent
		if (siteparentfound = 0 or newscopeparentneeded = 1 or scopeparentmoved = 1) then

			-- INSERT SYNTHETIC SiteNameParent EVENT
			insert into alerts.status (
				Identifier,
				Node,
				Class,
				Summary,
				AlertGroup,
				Severity,
				ScopeID,
				SiteName,
				FirstOccurrence,
				LastOccurrence,
				ParentIdentifier,
				Grade,
				OwnerUID,
				OwnerGID)
			values (
				siteidentifier,
				nodetouse,
				99990,
				new.SiteName + ': calculating summary details...',
				'SiteNameParent',
				1,
				scopeid,
				new.SiteName,
				now,
				now,
				scopeidentifier,
				1,
				65534,
				0);
		end if;

		-- LINK CURRENT EVENT TO SiteNameParent
		set new.ParentIdentifier = siteidentifier;
	end if;
end;
go

------------------------------------------------------------------------------
-- CREATE UPDATE TRIGGER ON alerts.status TO CREATE PARENT EVENTS AND LINK CHILDREN
------------------------------------------------------------------------------

CREATE OR REPLACE TRIGGER correlation_update
GROUP correlation_triggers
PRIORITY 15
COMMENT 'Checks for the existence of parent events, updates or creates if necessary'
BEFORE UPDATE ON alerts.status
FOR EACH ROW
WHEN get_prop_value('ActingPrimary') %= 'TRUE' and
	old.ScopeID = '' and new.ScopeID != '' and
	old.ParentIdentifier = '' and
	new.AlertGroup not in ('SiteNameParent', 'ScopeIDParent', 'Synthetic Event - Parent')
declare

	quietperiod		INTEGER;
	usenodeforscopeidparent	INTEGER;
	usenodeforsitenameparent INTEGER;
	nodetouse		CHAR(64);
	nositenameparentifsitenameblank INTEGER;

	now			TIME;
	expiretime		TIME;

	scopeid			CHAR(255);

	scopeparentfound	INTEGER;
	newscopeparentneeded	INTEGER;
	scopeidentifier		CHAR(255);

	siteparentfound		INTEGER;
	scopeparentmoved	INTEGER;
	siteidentifier		CHAR(255);

begin

-- STEP 1: INITIALISE VARIABLES

	set quietperiod = 15 * 60;
	set usenodeforscopeidparent = 0;
	set usenodeforsitenameparent = 0;
	set nodetouse = '';
	set nositenameparentifsitenameblank = 0;

	set now = getdate();

	set scopeid = '';

	set scopeparentfound = 0;
	set newscopeparentneeded = 0;
	set scopeidentifier = '';

	set siteparentfound = 0;
	set scopeparentmoved = 0;
	set siteidentifier = '';

	-- SET UP VARIABLES BASED ON PROPERTIES
	for each row property in master.properties where property.Name in (
		'SEGQuietPeriod',
		'SEGUseNodeForScopeIDParent',
		'SEGUseNodeForSiteNameParent',
		'SEGNoSiteNameParentIfSiteNameBlank')
	begin
		-- SPECIFIES THE GLOBAL DEFAULT QUIET PERIOD TO APPLY TO GROUPING
		if (property.Name = 'SEGQuietPeriod') then

			set quietperiod = property.IntValue;

		-- SPECIFIES IF THE NODE VALUE OF THE FIRST EVENT SHOULD BE USED FOR THE ScopeIDParent EVENT
		elseif (property.Name = 'SEGUseNodeForScopeIDParent') then

			set usenodeforscopeidparent = property.IntValue;

		-- SPECIFIES IF THE NODE VALUE OF THE FIRST EVENT SHOULD BE USED FOR THE SiteNameParent EVENT
		elseif (property.Name = 'SEGUseNodeForSiteNameParent') then

			set usenodeforsitenameparent = property.IntValue;

		-- SPECIFIES WHETHER OR NOT A SiteNameParent SHOULD BE CREATED IF SiteName IS NULL
		elseif (property.Name = 'SEGNoSiteNameParentIfSiteNameBlank') then

			set nositenameparentifsitenameblank = property.IntValue;

		end if;
	end;

	-- OVERRIDE QUIET PERIOD IF DEFINED IN THE INCOMING EVENT
	if (new.QuietPeriod != 0) then

		set quietperiod = new.QuietPeriod;
	end if;

	-- USE VALUE STORED IN FirstOccurrence IF SET, ELSE USE CURRENT TIME
	if (new.FirstOccurrence = 0) then

		set now = getdate();
	else 

		set now = new.FirstOccurrence;
	end if;

	-- SET DEFAULT VALUE OF ExpireTime BASED ON FirstOccurrence
	set expiretime = now + quietperiod;

-- STEP 2: CHECK ScopeAlias MEMBERSHIP

	-- CHECK FOR ScopeAlias MEMBERSHIP
	for each row scopealiasmemberrow in master.correlation_scopealias_members where
		scopealiasmemberrow.ScopeID = new.ScopeID
	begin

		-- SET THE ScopeID TO USE FOR THE SYNTHETIC CONTAINMENT EVENT TO BE THE ALIAS
		set scopeid = scopealiasmemberrow.ScopeAlias;
	end;

	-- USE INCOMING ScopeID VALUE IF ScopeID IS NOT A MEMBER OF AN ALIAS GROUP
	if (scopeid = '') then

		set scopeid = new.ScopeID;
	end if;

-- STEP 3: SET UP ScopeIDParent EVENT

	-- LOOK FOR CURRENT ScopeIDParent
	for each row parent in master.correlation_scopeid where
		parent.ScopeID = scopeid
	begin

		-- MARK ScopeID PARENT AS FOUND
		set scopeparentfound = 1;

		-- CHECK IF THE GROUP ExpireTime HAS PASSED
		if (parent.ExpireTime < now) then

			-- IF SO, UPDATE PARENT ENTRY AND SET FLAG TO CREATE A NEW ScopeIDParent
			set parent.Identifier =	'ScopeIDParent:' + parent.ScopeID + ':' + to_char(now);
			set newscopeparentneeded = 1;
		end if;

		-- STORE EITHER EXISTING OR UPDATED ScopeIDParent IDENTIFIER
		set scopeidentifier = parent.Identifier;

		-- UPDATE LastOccurrence IN PARENT ENTRY IF INCOMING VALUE IS GREATER
		if (parent.LastOccurrence < now) then

			set parent.LastOccurrence = now;
		end if;

		-- UPDATE GROUP ExpireTime IF INCOMING ROW EXTENDS IT
		if (parent.ExpireTime < expiretime) then

			set parent.ExpireTime = expiretime;
		end if;
	end;

	-- CREATE PARENT ENTRY IN master.correlation_scopeid IF NONE FOUND
	if (scopeparentfound = 0) then

		-- SET ScopeIDParent IDENTIFIER TO A NEW VALUE
		set scopeidentifier = 'ScopeIDParent:' + scopeid + ':' + to_char(now);

		-- INSERT NEW ScopeIDParent
		insert into master.correlation_scopeid (
			ScopeID,
			LastOccurrence,
			Identifier,
			ExpireTime)
		values (
			scopeid,
			now,
			scopeidentifier,
			expiretime);
	end if;

	-- SET THE NODE FIELD VALUE TO USE BASED ON PROPERTY
	if (usenodeforscopeidparent = 1) then

		set nodetouse = new.Node;
	else

		set nodetouse = scopeid;
	end if;

	-- INSERT ScopeIDParent EVENT INTO alerts.status IF NOT PRESENT OR NEW ONE NEEDED
	if (scopeparentfound = 0 or newscopeparentneeded = 1) then

		-- INSERT SYNTHETIC ScopeIDParent EVENT
		insert into alerts.status (
			Identifier,
			Node,
			Class,
			Summary,
			AlertGroup,
			Severity,
			ScopeID,
			FirstOccurrence,
			LastOccurrence,
			Grade,
			OwnerUID,
			OwnerGID)
		values (
			scopeidentifier,
			nodetouse,
			99990,
			'INCIDENT: ' + scopeid + ': calculating summary details...',
			'ScopeIDParent',
			1,
			scopeid,
			now,
			now,
			1,
			65534,
			0);
	end if;

-- STEP 4: SET UP SiteNameParent EVENT IF OPTION TO DISABLE IS NOT SET

	-- DO NOT CREATE SiteNameParent EVENT IF OPTION TO DISABLE SiteNameParent
	-- IS SET AND SiteName IS BLANK.  INSTEAD LINK EVENT TO ScopeIDParent EVENT
	if (nositenameparentifsitenameblank = 1 and new.SiteName = '') then

		-- LINK CURRENT EVENT TO ScopeIDParent EVENT
		set new.ParentIdentifier = scopeidentifier;

	-- ELSE CREATE SiteNameParent EVENT AND LINK CURRENT EVENT TO THAT
	else

		-- IF EVENT HAS NO SiteName THEN SET IT TO A DEFAULT
		if (new.SiteName = '') then

			set new.SiteName = 'NO SITENAME';
		end if;

		-- LOOK FOR CURRENT SiteNameParent
		for each row parent in master.correlation_sitename where
			parent.SiteName = new.SiteName and
			parent.ScopeID = scopeid
		begin

			-- MARK SiteName PARENT AS FOUND
			set siteparentfound = 1;

			-- CHECK IF ScopeIDParent HAS MOVED
			for each row scopeidparentevent in alerts.status where
				scopeidparentevent.Identifier = parent.Identifier
			begin

				if (scopeidparentevent.ParentIdentifier != scopeidentifier) then

					set scopeparentmoved = 1;
				end if;
			end;

			-- UPDATE THE ENTRY IF WE ARE CREATING A NEW ScopeIDParent
			if (newscopeparentneeded = 1 or scopeparentmoved = 1) then

				set parent.Identifier = 'SiteNameParent:' +
					scopeid + ':' + new.SiteName + ':' + to_char(now);
			end if;

			-- STORE CURRENT SiteName PARENT IDENTIFIER
			set siteidentifier = parent.Identifier;
		end;

		-- CREATE PARENT ENTRY IN master.correlation_sitename IF NONE FOUND
		if (siteparentfound = 0) then

			-- STORE NEW SiteName PARENT IDENTIFIER
			set siteidentifier = 'SiteNameParent:' +
				scopeid + ':' + new.SiteName + ':' + to_char(now);

			-- CREATE THE NEW SiteParent EVENT
			insert into master.correlation_sitename (
				SiteName,
				ScopeID,
				Identifier)
			values (
				new.SiteName,
				scopeid,
				siteidentifier);
		end if;

		-- SET THE NODE FIELD VALUE TO USE BASED ON PROPERTY
		if (usenodeforsitenameparent = 1) then

			set nodetouse = new.Node;
		else

			set nodetouse = new.SiteName;
		end if;

		-- INSERT NEW SiteParent EVENT INTO alerts.status IF NEW OR UPDATED ScopeIDParent
		if (siteparentfound = 0 or newscopeparentneeded = 1 or scopeparentmoved = 1) then

			-- INSERT SYNTHETIC SiteNameParent EVENT
			insert into alerts.status (
				Identifier,
				Node,
				Class,
				Summary,
				AlertGroup,
				Severity,
				ScopeID,
				SiteName,
				FirstOccurrence,
				LastOccurrence,
				ParentIdentifier,
				Grade,
				OwnerUID,
				OwnerGID)
			values (
				siteidentifier,
				nodetouse,
				99990,
				new.SiteName + ': calculating summary details...',
				'SiteNameParent',
				1,
				scopeid,
				new.SiteName,
				now,
				now,
				scopeidentifier,
				1,
				65534,
				0);
		end if;

		-- LINK CURRENT EVENT TO SiteNameParent
		set new.ParentIdentifier = siteidentifier;
	end if;
end;
go

------------------------------------------------------------------------------
-- CREATE DELETE TRIGGER ON alerts.status TO REMOVE PARENT ENTRIES
------------------------------------------------------------------------------

CREATE OR REPLACE TRIGGER correlation_delete_row
GROUP correlation_triggers
PRIORITY 15
COMMENT 'Deletes from the master.correlation_sitename or master.correlation_scopeid tables where needed'
AFTER DELETE ON alerts.status
FOR EACH ROW
WHEN get_prop_value('ActingPrimary') %= 'TRUE'
begin

	-- DELETE CORRESPONDING ENTRY IN MASTER CORRELATION TABLE FOR SiteNameParent EVENTS
	if (old.AlertGroup = 'SiteNameParent') then

		delete from master.correlation_sitename where Identifier = old.Identifier;

	-- DELETE CORRESPONDING ENTRY IN MASTER CORRELATION TABLE FOR ScopeIDParent EVENTS
	elseif (old.AlertGroup = 'ScopeIDParent') then

		delete from master.correlation_scopeid where Identifier = old.Identifier;

	end if;
end;
go

------------------------------------------------------------------------------
-- CREATE PROCEDURES TO PROCESS PARENT EVENTS
------------------------------------------------------------------------------

------------------------------------------------------------------------------
-- CREATE A PROCEDURE TO DISMANTLE EXPIRED GROUPS THAT ARE SMALLER THAN A GIVEN THRESHOLD

CREATE OR REPLACE PROCEDURE correlation_dismantle_small_groups ( )
declare

	minimumchildren INTEGER;
	minimumttl INTEGER;
	clearttnumber INTEGER;
	activescopeid INTEGER;

begin

	-- INITIALISE LOCAL VARIABLES
	set minimumchildren = 0;
	set minimumttl = 120;
	set clearttnumber = 0;
	set activescopeid = 0;

	-- LOAD UP VARIABLES BASED ON PROPERTIES
	for each row property in master.properties where property.Name in (
		'SEGMinimumChildren', 'SEGMinimumScopeIDTTL', 'SEGClearTTNumber')
	begin

		-- SPECIFIES THE MINIMUM NUMBER OF CHILDREN A ScopeIDParent MUST HAVE TO LIVE BEYOND ITS EXPIRETIME
		if (property.Name = 'SEGMinimumChildren') then

			set minimumchildren = property.IntValue;

		-- SPECIFIES THE MINIMUM TIME TO LIVE OF ScopeIDParent EVENTS
		elseif (property.Name = 'SEGMinimumScopeIDTTL') then

			set minimumttl = property.IntValue;

		-- SPECIFIES THAT INHERITED TICKET NUMBER SHOULD BE CLEARED FROM CHILDREN IF DISMANTLING
		elseif (property.Name = 'SEGClearTTNumber') then

			set clearttnumber = property.IntValue;
		end if;
	end;

	-- FOR EACH ScopeIDParent WITH LESS THAN THE THRESHOLD NUMBER OF CHILDREN
	for each row smallparent in alerts.status where
		smallparent.Grade < minimumchildren and
		smallparent.InternalLast < (getdate() - minimumttl) and
		smallparent.AlertGroup = 'ScopeIDParent' and
		smallparent.Identifier in (select ParentIdentifier from alerts.status)
	begin

		-- RESET ACTIVESCOPEID
		set activescopeid = 0;

		-- LOOK FOR A VALID ENTRY IN THE master.correlation_scopeid TABLE
		for each row scopeidentry in master.correlation_scopeid where
			scopeidentry.Identifier = smallparent.Identifier and
			scopeidentry.ExpireTime > getdate()
		begin

			-- MARK THIS AS AN ACTIVE ScopeID
			set activescopeid = 1;
		end;

		-- IF THE ScopeIDParent EVENT IS NOT ACTIVE, DISMANTLE IT
		if (activescopeid = 0) then

			-- UNLINK EACH CHILD EVENT
			for each row child in alerts.status where child.ParentIdentifier = smallparent.Identifier
			begin

				-- IF THE CHILD IS A SiteNameParent, UNLINK ITS CHILDREN
				if (child.AlertGroup = 'SiteNameParent') then

					for each row grandchild in alerts.status where 
						grandchild.ParentIdentifier = child.Identifier
					begin

						-- UNLINK GRANDCHILD FROM SiteNameParent
						set grandchild.ParentIdentifier = '';

						-- CLEAR CHILD INHERITED TTNumber IF OPTION TO CLEAR IS SET
						if (clearttnumber = 1 and
							grandchild.TTNumber != '' and
							grandchild.TTNumber = child.TTNumber) then

							set grandchild.TTNumber = '';
						end if;
					end;
				end if;

				-- UNLINK CHILD FROM ScopeIDParent
				set child.ParentIdentifier = '';

				-- CLEAR CHILD INHERITED TTNumber IF OPTION TO CLEAR IS SET
				if (clearttnumber = 1 and
					child.TTNumber != '' and
					child.TTNumber = smallparent.TTNumber) then

					set child.TTNumber = '';
				end if;
			end;
		end if;
	end;
end;
go

------------------------------------------------------------------------------
-- CREATE A PROCEDURE TO CLEAR EXPIRED PARENT EVENTS

CREATE OR REPLACE PROCEDURE correlation_clear_expired_parents ( )
declare

	parentfound INTEGER;

begin

	-- INITIALISE VARIABLES
	set parentfound = 0;

	-- TARGET PARENT EVENTS THAT ARE NOT CLEAR AND THAT HAVE NO CHILDREN EVENTS
	for each row parent in alerts.status where
		parent.Severity != 0 and
		parent.AlertGroup in ('SiteNameParent', 'ScopeIDParent') and
		parent.Identifier not in (select ParentIdentifier from alerts.status)
	begin

		-- MARK GROUPING ENTRY AS NOT FOUND INITIALLY
		set parentfound = 0;

		-- LOOK IN THE master.correlation_scopeid TABLE TO SEE IF THE ScopeID
		-- GROUPING HAS EXPIRED.  IF IT IS NOT FOUND, IT HAS ALREADY EXPIRED.
		for each row scopeidentry in master.correlation_scopeid where
			scopeidentry.ScopeID = parent.ScopeID
		begin

			-- IF CURRENT EVENT IS A SiteNameParent EVENT, CHECK THAT THE ACTIVE ScopeIDParent
			-- PRESENT IS ACTUALLY ITS DIRECT PARENT.  IF IT IS NOT, THEN MARK IT AS NOT FOUND.
			if (parent.AlertGroup = 'SiteNameParent' and
				parent.ParentIdentifier != scopeidentry.Identifier) then

				set parentfound = 0;

			-- ELSE A VALID PARENT HAS BEEN FOUND
			else

				-- MARK ScopeID ENTRY AS HAVING BEEN FOUND
				set parentfound = 1;

				-- CHECK IF ScopeID HAS EXPIRED
				if (scopeidentry.ExpireTime < getdate()) then

					-- CLEAR PARENT EVENT DUE TO GROUP EXPIRATION
					set parent.Severity = 0;
				end if;
			end if;
		end;

		-- IF ScopeID GROUPING IS NOT FOUND, CLEAR IT AS IT HAS ALREADY EXPIRED
		if (parentfound = 0) then

			set parent.Severity = 0;
		end if;
	end;
end;
go

------------------------------------------------------------------------------
-- CREATE A PROCEDURE TO PROCESS SiteNameParent EVENTS

CREATE OR REPLACE PROCEDURE correlation_process_sitenameparents ( )
declare

	propagatettnumber INTEGER;
	propagateacknowledged INTEGER;
	propagateowneruid INTEGER;
	propagateownergid INTEGER;

	usesitenameprefix INTEGER;
	sitenameprefix CHAR(255);
	usesitenamelabel INTEGER;
	usesitenameimpactcause INTEGER;
	usesitenamecustomtext INTEGER;
	propagatetexttositenameparentcause INTEGER;
	propagatetexttositenameparentimpact INTEGER;
	propagatetexttositenameparentfirst INTEGER;
	propagatetexttositenameparentlast INTEGER;
	customtext CHAR(255);
	usesitenamenumactivealarms INTEGER;

	highcauseweight INTEGER;
	highcausetext CHAR(255);
	highimpactweight INTEGER;
	highimpacttext CHAR(255);
	highseverity INTEGER;
	lowfirstoccurrence TIME;
	highlastoccurrence TIME;
	alarmcounter INTEGER;
	summary CHAR(255);

	journaluid INTEGER;
	journaltoscopeidparent INTEGER;
	scopeidserial INTEGER;
	scopeidjournalcount INTEGER;
	journaltositenameparent INTEGER;
	journalmaxeventsperentry INTEGER;
	journalservernameserverserial INTEGER;
	journalnode INTEGER;
	journalsummary INTEGER;
	journalalertkey INTEGER;
	journalcustomtext INTEGER;
	maxsitenamejournals INTEGER;
	maxscopeidjournals INTEGER;
	thisjournalentry CHAR(4080);
	sitenameparentjournalentry CHAR(4080);
	scopeidparentjournalentry CHAR(4080);
	sitenamejournalcounter INTEGER;
	scopeidjournalcounter INTEGER;

begin

	-- INITIALISE LOCAL VARIABLES
	set propagatettnumber = 1;
	set propagateacknowledged = 1;
	set propagateowneruid = 1;
	set propagateownergid = 1;

	set usesitenameprefix = 0;
	set sitenameprefix = '';
	set usesitenamelabel = 1;
	set usesitenameimpactcause = 1;
	set usesitenamecustomtext = 0;
	set propagatetexttositenameparentcause = 0;
	set propagatetexttositenameparentimpact = 0;
	set propagatetexttositenameparentfirst = 0;
	set propagatetexttositenameparentlast = 0;
	set customtext = '';
	set usesitenamenumactivealarms = 1;

	set highcauseweight = 0;
	set highcausetext = 'UNKNOWN';
	set highimpactweight = 0;
	set highimpacttext  = 'UNKNOWN';
	set highseverity = 0;
	set lowfirstoccurrence = 0;
	set highlastoccurrence = 0;
	set alarmcounter  = 0;
	set summary = '';

	set journaluid = 0;
	set journaltoscopeidparent = 0;
	set scopeidserial = 0;
	set scopeidjournalcount = 0;
	set journaltositenameparent = 0;
	set journalmaxeventsperentry = 20;
	set journalservernameserverserial = 1;
	set journalnode = 1;
	set journalsummary = 1;
	set journalalertkey = 0;
	set journalcustomtext = 0;
	set maxsitenamejournals = 10;
	set maxscopeidjournals = 50;
	set thisjournalentry = '';
	set sitenameparentjournalentry = '';
	set scopeidparentjournalentry = '';
	set sitenamejournalcounter = 0;
	set scopeidjournalcounter = 0;

	-- LOAD UP VARIABLES BASED ON PROPERTIES
	for each row property in master.properties where property.Name in (
		'SEGPropagateTTNumber',
		'SEGPropagateAcknowledged',
		'SEGPropagateOwnerUID',
		'SEGPropagateOwnerGID',
		'SEGUseSiteNamePrefix',
		'SEGSiteNamePrefix',
		'SEGUseSiteNameLabel',
		'SEGUseSiteNameImpactCause',
		'SEGUseSiteNameCustomText',
		'SEGPropagateTextToSiteNameParentCause',
		'SEGPropagateTextToSiteNameParentImpact',
		'SEGPropagateTextToSiteNameParentFirst',
		'SEGPropagateTextToSiteNameParentLast',
		'SEGUseSiteNameNumActiveAlarms',
		'SEGJournalUID',
		'SEGJournalToSiteNameParent',
		'SEGJournalToScopeIDParent',
		'SEGJournalMaxEventsPerEntry',
		'SEGJournalServerNameServerSerial',
		'SEGJournalNode',
		'SEGJournalSummary',
		'SEGJournalAlertKey',
		'SEGJournalCustomText',
		'SEGMaxSiteNameJournals',
		'SEGMaxScopeIDJournals')
	begin

		-- SPECIFIES WHETHER OR NOT TO PROPAGATE THE TTNumber TO CHILD EVENTS
		if (property.Name = 'SEGPropagateTTNumber') then

			set propagatettnumber = property.IntValue;

		-- SPECIFIES WHETHER OR NOT TO PROPAGATE THE Acknowledged STATUS TO CHILD EVENTS
		elseif (property.Name = 'SEGPropagateAcknowledged') then

			set propagateacknowledged = property.IntValue;

		-- SPECIFIES WHETHER OR NOT TO PROPAGATE THE OwnerUID TO CHILD EVENTS
		elseif (property.Name = 'SEGPropagateOwnerUID') then

			set propagateowneruid = property.IntValue;

		-- SPECIFIES WHETHER OR NOT TO PROPAGATE THE OwnerGID TO CHILD EVENTS
		elseif (property.Name = 'SEGPropagateOwnerGID') then

			set propagateownergid = property.IntValue;

		-- SPECIFIES WHETHER OR NOT TO INCLUDE THE SiteName PREFIX IN THE SiteNameParent EVENT
		elseif (property.Name = 'SEGUseSiteNamePrefix') then

			set usesitenameprefix = property.IntValue;

		-- SPECIFIES THE SiteName PREFIX TO USE IN THE SiteNameParent EVENT
		elseif (property.Name = 'SEGSiteNamePrefix') then

			set sitenameprefix = property.CharValue;

		-- SPECIFIES WHETHER OR NOT TO INCLUDE THE ACTUAL SiteName IN THE SiteNameParent EVENT
		elseif (property.Name = 'SEGUseSiteNameLabel') then

			set usesitenamelabel = property.IntValue;

		-- SPECIFIES WHETHER OR NOT TO INCLUDE THE IMPACT AND CAUSE INFO IN THE SiteNameParent EVENT
		elseif (property.Name = 'SEGUseSiteNameImpactCause') then

			set usesitenameimpactcause = property.IntValue;

		-- SPECIFIES WHETHER OR NOT TO INCLUDE THE CUSTOM TEXT IN THE SiteNameParent EVENT
		elseif (property.Name = 'SEGUseSiteNameCustomText') then

			set usesitenamecustomtext = property.IntValue;

		-- SPECIFIES WHETHER OR NOT TO PROPAGATE THE CUSTOM TEXT TO THE ScopeIDParent EVENT
		-- BASED ON THE CUSTOM TEXT OF THE CHILD EVENT WITH THE HIGHEST CauseWeight
		elseif (property.Name = 'SEGPropagateTextToSiteNameParentCause') then

			set propagatetexttositenameparentcause = property.IntValue;

		-- SPECIFIES WHETHER OR NOT TO PROPAGATE THE CUSTOM TEXT TO THE ScopeIDParent EVENT
		-- BASED ON THE CUSTOM TEXT OF THE CHILD EVENT WITH THE HIGHEST ImpactWeight
		elseif (property.Name = 'SEGPropagateTextToSiteNameParentImpact') then

			set propagatetexttositenameparentimpact = property.IntValue;

		-- SPECIFIES WHETHER OR NOT TO PROPAGATE THE CUSTOM TEXT TO THE ScopeIDParent EVENT
		-- BASED ON THE CUSTOM TEXT OF THE CHILD EVENT WITH THE LOWEST FirstOccurrence
		elseif (property.Name = 'SEGPropagateTextToSiteNameParentFirst') then

			set propagatetexttositenameparentfirst = property.IntValue;

		-- SPECIFIES WHETHER OR NOT TO PROPAGATE THE CUSTOM TEXT TO THE ScopeIDParent EVENT
		-- BASED ON THE CUSTOM TEXT OF THE CHILD EVENT WITH THE HIGHEST LastOccurrence
		elseif (property.Name = 'SEGPropagateTextToSiteNameParentLast') then

			set propagatetexttositenameparentlast = property.IntValue;

		-- SPECIFIES WHETHER OR NOT TO INCLUDE THE NUMBER OF ACTIVE ALARMS IN THE SiteNameParent EVENT
		elseif (property.Name = 'SEGUseSiteNameNumActiveAlarms') then

			set usesitenamenumactivealarms = property.IntValue;

		-- SPECIFIES THE UID TO USE WHEN INSERTING JOURNALS INTO SYNTHETIC PARENT EVENTS - DEFAULT IS ROOT
		elseif (property.Name = 'SEGJournalUID') then

			set journaluid = property.IntValue;

		-- SPECIFIES WHETHER OR NOT TO JOURNAL CHILD EVENTS TO THE ScopeIDParent EVENT
		elseif (property.Name = 'SEGJournalToScopeIDParent') then

			set journaltoscopeidparent = property.IntValue;

		-- SPECIFIES WHETHER OR NOT TO JOURNAL CHILD EVENTS TO THE SiteNameParent EVENT
		elseif (property.Name = 'SEGJournalToSiteNameParent') then

			set journaltositenameparent = property.IntValue;

		-- SPECIFIES HOW MANY EVENTS TO ROLL UP INTO A SINGLE JOURNAL ENTRY
		elseif (property.Name = 'SEGJournalMaxEventsPerEntry') then

			set journalmaxeventsperentry = property.IntValue;

		-- SPECIFIES WHETHER OR NOT TO SEND THE ServerName:ServerSerial FIELDS TO THE PARENT EVENT JOURNAL
		elseif (property.Name = 'SEGJournalServerNameServerSerial') then

			set journalservernameserverserial = property.IntValue;

		-- SPECIFIES WHETHER OR NOT TO SEND THE Node FIELD TO THE PARENT EVENT JOURNAL
		elseif (property.Name = 'SEGJournalNode') then

			set journalnode = property.IntValue;

		-- SPECIFIES WHETHER OR NOT TO SEND THE Summary FIELD TO THE PARENT EVENT JOURNAL
		elseif (property.Name = 'SEGJournalSummary') then

			set journalsummary = property.IntValue;

		-- SPECIFIES WHETHER OR NOT TO SEND THE AlertKey FIELD TO THE PARENT EVENT JOURNAL
		elseif (property.Name = 'SEGJournalAlertKey') then

			set journalalertkey = property.IntValue;

		-- SPECIFIES WHETHER OR NOT TO SEND THE CustomText FIELD TO THE PARENT EVENT JOURNAL
		elseif (property.Name = 'SEGJournalCustomText') then

			set journalcustomtext = property.IntValue;

		-- SPECIFIES MAXIMUM NUMBER OF EVENTS TO SEND TO THE SiteNameParent EVENT JOURNAL
		elseif (property.Name = 'SEGMaxSiteNameJournals') then

			set maxsitenamejournals = property.IntValue;

		-- SPECIFIES MAXIMUM NUMBER OF EVENTS TO SEND TO THE ScopeIDParent EVENT JOURNAL
		elseif (property.Name = 'SEGMaxScopeIDJournals') then

			set maxscopeidjournals = property.IntValue;
		end if;
	end;

	-- IN CASE MULTIPLE CustomText PROPAGATION OPTIONS ARE INADVERTENTLY SELECTED,
	-- ENFORCE AN ORDER OF PRECEDENCE TO ENSURE UNDEFINED RESULTS ARE AVOIDED
	--
	-- ORDER OF PRECEDENCE IS:
	-- 1 PROPAGATE CustomText OF HIGHEST CauseWeight
	-- 2 PROPAGATE CustomText OF HIGHEST ImpactWeight
	-- 3 PROPAGATE CustomText OF LOWEST FirstOccurrence
	-- 4 PROPAGATE CustomText OF HIGHEST LastOccurrence
	--
	if (propagatetexttositenameparentcause = 1) then

		-- CLEAR OTHER SELECTED OPTIONS
		set propagatetexttositenameparentimpact = 0;
		set propagatetexttositenameparentfirst = 0;
		set propagatetexttositenameparentlast = 0;

	elseif (propagatetexttositenameparentimpact = 1) then

		-- CLEAR OTHER SELECTED OPTIONS
		set propagatetexttositenameparentfirst = 0;
		set propagatetexttositenameparentlast = 0;

	elseif (propagatetexttositenameparentfirst = 1) then

		-- CLEAR OTHER SELECTED OPTIONS
		set propagatetexttositenameparentlast = 0;

	end if;

	-- UPDATE EACH SiteNameParent EVENT
	for each row site in alerts.status where site.AlertGroup = 'SiteNameParent'
	begin

		-- RESET LOCAL VARIABLES
		set highcauseweight = 0;
		set highcausetext = 'UNKNOWN';
		set highimpactweight = 0;
		set highimpacttext  = 'UNKNOWN';
		set highseverity = 0;
		set lowfirstoccurrence = 0;
		set highlastoccurrence = 0;
		set customtext = '';
		set alarmcounter  = 0;
		set summary = '';
		set scopeidserial = 0;
		set thisjournalentry = '';
		set sitenameparentjournalentry = '';
		set scopeidparentjournalentry = '';
		set sitenamejournalcounter = 0;
		set scopeidjournalcounter = 0;

		-- IF WE ARE SENDING JOURNALS TO THE ScopeIDParent, WE NEED ITS SERIAL
		-- AND CURRENT JOURNAL COUNT
		if (journaltoscopeidparent = 1) then

			-- LOOK UP ScopeIDParent
			for each row scopeidparent in alerts.status where
				scopeidparent.Identifier = site.ParentIdentifier
			begin

				set scopeidserial = scopeidparent.Serial;
				set scopeidjournalcount = scopeidparent.Poll;
			end;
		end if;

		-- EXAMINE EACH CHILD OF THE CURRENT SiteNameParent EVENT
		for each row child in alerts.status where
			child.ParentIdentifier = site.Identifier
		begin

			-- ONLY CONSIDER NON-CLEARED CHILD EVENTS
			if (child.Severity > 0) then

				-- INCREMENT A COUNTER FOR THE NUMBER OF CHILDREN
				set alarmcounter = alarmcounter + 1;

				-- STORE THE HIGHEST Severity OF THE CHILDREN EVENTS
				if (highseverity < child.Severity) then

					set highseverity = child.Severity;
				end if;

				-- STORE THE HIGHEST CauseWeight OF THE CHILDREN EVENTS
				-- AND ITS ASSOCIATED NormalisedAlarmName
				if (child.CauseWeight > highcauseweight) then
	
					set highcauseweight = child.CauseWeight;
					set highcausetext = child.NormalisedAlarmName;

					-- IF OPTION TO PROPAGATE CUSTOM TEXT TO SiteNameParent EVENT FROM HIGHEST
					-- CauseWeight CHILD IS ENABLED, STORE CURRENT EVENT CUSTOM TEXT
					if (propagatetexttositenameparentcause = 1) then

						set customtext = child.CustomText;
					end if;
				end if;

				-- STORE THE HIGHEST ImpactWeight OF THE CHILDREN EVENTS
				-- AND ITS ASSOCIATED NormalisedAlarmName
				if (child.ImpactWeight > highimpactweight) then
	
					set highimpactweight = child.ImpactWeight;
					set highimpacttext = child.NormalisedAlarmName;

					-- IF OPTION TO PROPAGATE CUSTOM TEXT TO SiteNameParent EVENT FROM HIGHEST
					-- ImpactWeight CHILD IS ENABLED, STORE CURRENT EVENT CUSTOM TEXT
					if (propagatetexttositenameparentimpact = 1) then

						set customtext = child.CustomText;
					end if;
				end if;

				-- STORE THE LOWEST NON-ZERO FirstOccurrence OF THE CHILDREN EVENTS
				if (lowfirstoccurrence > child.FirstOccurrence or lowfirstoccurrence = 0) then

					set lowfirstoccurrence = child.FirstOccurrence;

					-- IF OPTION TO PROPAGATE CUSTOM TEXT TO SiteNameParent EVENT FROM LOWEST
					-- FirstOccurrence CHILD IS ENABLED, STORE CURRENT EVENT CUSTOM TEXT
					if (propagatetexttositenameparentfirst = 1) then

						set customtext = child.CustomText;
					end if;
				end if;

				-- STORE THE HIGHEST LastOccurrence OF THE CHILDREN EVENTS
				if (highlastoccurrence < child.LastOccurrence) then
	
					set highlastoccurrence = child.LastOccurrence;

					-- IF OPTION TO PROPAGATE CUSTOM TEXT TO SiteNameParent EVENT FROM HIGHEST
					-- LastOccurrence CHILD IS ENABLED, STORE CURRENT EVENT CUSTOM TEXT
					if (propagatetexttositenameparentlast = 1) then

						set customtext = child.CustomText;
					end if;
				end if;

				-- PROPAGATE THE TICKET NUMBER FROM THE SiteNameParent EVENT TO
				-- THE CURRENT CHILD IF THE CURRENT CHILD IS UNTICKETED,
				-- UNSUPPRESSED AND THE OPTION TO PROPAGATE TICKET NUMBER IS ENABLED
				if (site.TTNumber != '' and child.TTNumber = '' and
					child.SuppressEscl != 4 and propagatettnumber = 1) then
	
					set child.TTNumber = site.TTNumber;
				end if;

				-- PROPAGATE THE ACKNOWLEDGED STATUS FROM THE SiteNameParent EVENT TO
				-- THE CURRENT CHILD IF THE CURRENT CHILD IS UNACKNOWLEDGED AND THE
				-- OPTION TO PROPAGATE ACKNOWLEDGED STATUS IS ENABLED
				if (site.Acknowledged = 1 and child.Acknowledged != 1 and
					propagateacknowledged = 1) then

					set child.Acknowledged = site.Acknowledged;
				end if;

				-- PROPAGATE THE OwnerUID FROM THE SiteNameParent EVENT TO THE
				-- CURRENT CHILD EVENT IF THE OWNER OF THE SiteNameParent EVENT
				-- IS DIFFERENT TO THAT OF THE CHILD EVENT AND THE
				-- OPTION TO PROPAGATE OwnerUID IS ENABLED
				if (site.OwnerUID != child.OwnerUID and propagateowneruid = 1) then

					set child.OwnerUID = site.OwnerUID;
				end if;

				-- PROPAGATE THE OwnerGID FROM THE SiteNameParent EVENT TO THE
				-- CURRENT CHILD EVENT IF THE GROUP OF THE SiteNameParent EVENT
				-- IS DIFFERENT TO THAT OF THE CHILD EVENT AND THE
				-- OPTION TO PROPAGATE OwnerGID IS ENABLED
				if (site.OwnerGID != child.OwnerGID and propagateownergid = 1) then

					set child.OwnerGID = site.OwnerGID;
				end if;

				-- CLEAR thisjournalentry VARIABLE
				set thisjournalentry = '';

				-- CONSTRUCT A JOURNAL ENTRY FOR CURRENT EVENT IF
				-- - THE CURRENT EVENT HAS NOT ALREADY BEEN JOURNALED
				-- - IF THE MAXIMUM NUMBER OF EVENTS PER JOURNAL ENTRY HAS NOT BEEN EXCEEDED
				-- - IF SENDING CHILD EVENT DATA AS JOURNALS TO SiteNameParent IS ENABLED OR
				-- - IF SENDING CHILD EVENT DATA AS JOURNALS TO ScopeIDParent IS ENABLED
				if (child.JournalSent = 0 and
					sitenamejournalcounter < journalmaxeventsperentry and
					scopeidjournalcounter < journalmaxeventsperentry and
					(journaltositenameparent = 1 OR journaltoscopeidparent = 1)) then

					-- CHECK WHETHER OR NOT TO INCLUDE THE ServerName:ServerSerial IN THIS JOURNAL ENTRY
					if (journalservernameserverserial = 1) then

						set thisjournalentry = thisjournalentry + child.ServerName +
							':' + to_char(child.ServerSerial);
					end if;

					-- CHECK WHETHER OR NOT TO INCLUDE THE Node IN THIS JOURNAL ENTRY
					if (journalnode = 1 and child.Node != '') then

						-- ADD COLON BEFORE APPENDING TEXT TO THIS JOURNAL ENTRY
						if (thisjournalentry != '') then
							set thisjournalentry = thisjournalentry + ': ';
						end if;

						set thisjournalentry = thisjournalentry + child.Node;
					end if;

					-- CHECK WHETHER OR NOT TO INCLUDE THE Summary IN THIS JOURNAL ENTRY
					if (journalsummary = 1 and child.Summary != '') then

						-- ADD COLON BEFORE APPENDING TEXT TO THIS JOURNAL ENTRY
						if (thisjournalentry != '') then
							set thisjournalentry = thisjournalentry + ': ';
						end if;

						set thisjournalentry = thisjournalentry + child.Summary;
					end if;

					-- CHECK WHETHER OR NOT TO INCLUDE THE AlertKey IN THIS JOURNAL ENTRY
					if (journalalertkey = 1 and child.AlertKey != '') then

						-- ADD COLON BEFORE APPENDING TEXT TO THIS JOURNAL ENTRY
						if (thisjournalentry != '') then
							set thisjournalentry = thisjournalentry + ': ';
						end if;

						set thisjournalentry = thisjournalentry + child.AlertKey;
					end if;

					-- CHECK WHETHER OR NOT TO INCLUDE THE CustomText IN THIS JOURNAL ENTRY
					if (journalcustomtext = 1 and child.CustomText != '') then

						-- ADD COLON BEFORE APPENDING TEXT TO THIS JOURNAL ENTRY
						if (thisjournalentry != '') then
							set thisjournalentry = thisjournalentry + ': ';
						end if;

						set thisjournalentry = thisjournalentry + child.CustomText;
					end if;
				end if;

				-- ADD CURRENT EVENT TO SiteNameParent JOURNAL ENTRY IF:
				-- - IF SENDING CHILD EVENT DATA AS JOURNALS TO SiteNameParent IS ENABLED
				-- - IF THE MAXIMUM NUMBER OF EVENTS FOR THIS SiteNameParent HAS NOT BEEN EXCEEDED
				-- - THE CURRENT JOURNAL ENTRY IS NOT NULL
				if (journaltositenameparent = 1 and
					(site.Poll + sitenamejournalcounter) < maxsitenamejournals and
					thisjournalentry != '') then

					-- APPEND A NEW LINE CHARACTER IF THERE IS ALREADY TEXT TO ADD
					if (sitenameparentjournalentry != '') then
						set sitenameparentjournalentry = sitenameparentjournalentry + '\n';
					end if;

					-- ADD THE CURRENT JOURNAL ENTRY TO THE EXISTING
					set sitenameparentjournalentry = sitenameparentjournalentry + thisjournalentry;

					-- INCREMENT THE SiteNameParent JOURNAL COUNTER
					set sitenamejournalcounter = sitenamejournalcounter + 1;

					-- MARK CHILD EVENT AS HAVING BEEN SENT
					set child.JournalSent = 1;
 				end if;

				-- ADD CURRENT EVENT TO ScopeIDParent JOURNAL ENTRY IF:
				-- - IF SENDING CHILD EVENT DATA AS JOURNALS TO ScopeIDParent IS ENABLED
				-- - IF THE MAXIMUM NUMBER OF EVENTS FOR THIS ScopeIDParent HAS NOT BEEN EXCEEDED
				-- - THE CURRENT JOURNAL ENTRY IS NOT NULL
				if (journaltoscopeidparent = 1 and
					(scopeidjournalcount + scopeidjournalcounter) < maxscopeidjournals and
					thisjournalentry != '') then

					-- APPEND A NEW LINE CHARACTER IF THERE IS ALREADY TEXT TO ADD
					if (scopeidparentjournalentry != '') then
						set scopeidparentjournalentry = scopeidparentjournalentry + '\n';
					end if;

					-- ADD THE CURRENT JOURNAL ENTRY TO THE EXISTING
					set scopeidparentjournalentry = scopeidparentjournalentry + thisjournalentry;

					-- INCREMENT THE SiteNameParent JOURNAL COUNTER
					set scopeidjournalcounter = scopeidjournalcounter + 1;

					-- MARK CHILD EVENT AS HAVING BEEN SENT
					set child.JournalSent = 1;
 				end if;
			end if;
		end;

		-- STORE THE NUMBER OF ACTIVE CHILD EVENTS IN THE GRADE FIELD, IF NOT ZERO
		if (alarmcounter != 0) then

			set site.Grade = alarmcounter;

		-- ELSE THE COUNT OF ACTIVE CHILD EVENTS IS ZERO, SO RESET GRADE FIELD TO ZERO
		elseif (site.Grade != 0) then

			set site.Grade = 0;
		end if;

		-- BEGIN THE CONSTRUCTION OF THE SiteNameParent Summary FIELD

		-- SPECIFY WHETHER TO INCLUDE THE SiteName PREFIX
		if (usesitenameprefix = 1) then

			set summary = sitenameprefix;
		end if;

                -- SPECIFY WHETHER TO INCLUDE THE SiteName LABEL
                if (usesitenamelabel = 1) then

                        -- APPEND A COLON IF summary NOT BLANK
                        if (summary != '') then
                                set summary = summary + ': ';
                        end if;

			set summary = summary + site.SiteName;
		end if;

		-- SPECIFY WHETHER TO INCLUDE THE CAUSE AND IMPACT TEXT
		if (usesitenameimpactcause = 1) then

			-- APPEND A COLON IF summary NOT BLANK
			if (summary != '') then
				set summary = summary + ': ';
			end if;

			-- APPEND CAUSE AND IMPACT TEXT
			if (highcausetext = 'UNKNOWN' and highimpacttext = 'UNKNOWN') then
				set summary = summary + 'CAUSE AND IMPACT: UNKNOWN';
			elseif (highcausetext = 'UNKNOWN' and highimpacttext != 'UNKNOWN') then
				set summary = summary + highimpacttext + ' UNKNOWN CAUSE';
			elseif (highcausetext != 'UNKNOWN' and highimpacttext = 'UNKNOWN') then
				set summary = summary + highcausetext + ' UNKNOWN IMPACT';
			elseif (highcausetext != 'UNKNOWN' and highimpacttext = highcausetext) then
				set summary = summary + 'CAUSE AND IMPACT: ' + highcausetext;
			else
				set summary = summary + highimpacttext + ' caused by ' + highcausetext;
			end if;
		end if;

		-- SPECIFY WHETHER TO INCLUDE THE CUSTOM SUMMARY TEXT CustomText
		if (usesitenamecustomtext = 1 and site.CustomText != '') then

			-- APPEND A COLON IF summary NOT BLANK
			if (summary != '') then
				set summary = summary + ': ';
			end if;

			-- APPEND CustomText
			set summary = summary + site.CustomText;
		end if;

		-- SPECIFY WHETHER TO INCLUDE THE NUMBER OF ACTIVE ALARMS IN THE GROUPING
		if (usesitenamenumactivealarms = 1) then

			-- APPEND A SPACE CHARACTER IF summary NOT BLANK
			if (summary != '') then
				set summary = summary + ' ';
			end if;

			if (alarmcounter = 1) then
				set summary = summary + '(' + to_char(alarmcounter) + ' active alarm)';
			else
				set summary = summary + '(' + to_char(alarmcounter) + ' active alarms)';
			end if;
		end if;

		-- ONLY UPDATE THE SiteNameParent EVENT IF IT IS NOT CLEAR
		-- OR THERE ARE ACTIVE CHILDREN PRESENT
		if (site.Severity != 0 or alarmcounter != 0) then

			-- UPDATE Summary FIELD WITH THE CONSTRUCTED summary FROM ABOVE
			set site.Summary = summary;

			-- UPDATE Severity WITH INDETERMINITE IF ALL ITS CHILDREN ARE CLEAR OR
			-- THE HIGHEST SEVERITY OF ITS CHILDREN
			if (highseverity = 0) then
				set site.Severity = 1;
			else
				set site.Severity = highseverity;
			end if;

			-- UPDATE FirstOccurrence WITH THE EARLIEST VALUE FROM ITS CHILDREN
			if (lowfirstoccurrence != 0) then
				set site.FirstOccurrence = lowfirstoccurrence;
			end if;

			-- UPDATE LastOccurrence WITH THE LATEST VALUE FROM ITS CHILDREN
			if (highlastoccurrence != 0) then
				set site.LastOccurrence = highlastoccurrence;
			end if;

			-- UPDATE CustomText FIELD OF THE PARENT IF ONE OF THE PROPAGATION
			-- FROM CHILD OPTIONS ARE ENABLED
			if (propagatetexttositenameparentcause = 1 or
				propagatetexttositenameparentimpact = 1 or
				propagatetexttositenameparentfirst = 1 or
				propagatetexttositenameparentlast = 1) then

				set site.CustomText = customtext;
			end if;
		end if;

		-- ADD A JOURNAL ENTRY TO THE CURRENT SiteNameParent IF APPLICABLE
		if (journaltositenameparent = 1 and sitenameparentjournalentry != '') then

			-- ADD THE ROLLED UP CHILDREN EVENTS TO THE CURRENT SiteNameParent AS A JOURNAL ENTRY
			EXECUTE jinsert(site.Serial, journaluid, getdate(), 'CHILD EVENTS:\n' +
				sitenameparentjournalentry);

			-- UPDATE THE NUMBER OF CHILD EVENTS APPENDED TO THIS PARENT EVENT
			set site.Poll = site.Poll + sitenamejournalcounter;

			-- APPEND NOTE THAT NO FURTHER JOURNALS WILL BE APPENDED TO THIS SiteNameParent
			-- DUE TO THE MAXIMUM NUMBER HAVING BEEN REACHED
			if (site.Poll = maxsitenamejournals) then

				EXECUTE jinsert(site.Serial, journaluid, getdate() + 1,
					'*** MAXIMUM OF ' + to_char(maxsitenamejournals) +
					' EVENTS HAVE BEEN JOURNALED TO THIS PARENT');
			end if;
		end if;

		-- ADD A JOURNAL ENTRY TO THE CURRENT ScopeIDParent IF APPLICABLE
		if (journaltoscopeidparent = 1 and scopeidparentjournalentry != '') then

			-- ADD THE ROLLED UP CHILDREN EVENTS TO THE CURRENT SiteNameParent AS A JOURNAL ENTRY
			EXECUTE jinsert(scopeidserial, journaluid, getdate() + scopeidjournalcount + 1,
				'SUB-GROUPING CHILD EVENTS: ' + site.SiteName + '\n' + scopeidparentjournalentry);

			-- UPDATE THE NUMBER OF CHILD EVENTS APPENDED TO THIS PARENT EVENT
			update alerts.status set Poll = scopeidjournalcount + scopeidjournalcounter where
					Identifier = site.ParentIdentifier;

			-- APPEND NOTE THAT NO FURTHER JOURNALS WILL BE APPENDED TO THIS ScopeIDParent
			-- DUE TO THE MAXIMUM NUMBER HAVING BEEN REACHED
			if ((scopeidjournalcount + scopeidjournalcounter) = maxscopeidjournals) then

				EXECUTE jinsert(scopeidserial, journaluid, getdate() + scopeidjournalcount + 2,
					'*** MAXIMUM OF ' + to_char(maxscopeidjournals) +
					' EVENTS HAVE BEEN JOURNALED TO THIS PARENT');
			end if;
		end if;
	end;
end;
go

------------------------------------------------------------------------------
-- CREATE A PROCEDURE TO PROCESS ScopeIDParent EVENTS

CREATE OR REPLACE PROCEDURE correlation_process_scopeidparents ( )
declare

	propagatettnumber INTEGER;
	propagateacknowledged INTEGER;
	propagateowneruid INTEGER;
	propagateownergid INTEGER;

	usescopeidprefix INTEGER;
	scopeidprefix CHAR(255);
	usescopeidlabel INTEGER;
	usescopeidimpactcause INTEGER;
	usescopeidcustomtext INTEGER;
	propagatetexttoscopeidparentcause INTEGER;
	propagatetexttoscopeidparentimpact INTEGER;
	propagatetexttoscopeidparentfirst INTEGER;
	propagatetexttoscopeidparentlast INTEGER;
	customtext CHAR(255);
	usescopeidsitesaffected INTEGER;
	usescopeidnumactivealarms INTEGER;

	highcauseweight INTEGER;
	highcausetext CHAR(255);
	highimpactweight INTEGER;
	highimpacttext CHAR(255);
	highseverity INTEGER;
	lowfirstoccurrence TIME;
	highlastoccurrence TIME;
	alarmcounter INTEGER;
	alarmcounter2 INTEGER;
	alarmcounter3 INTEGER;
	summary CHAR(255);

	journaluid INTEGER;
	journaltoscopeidparent INTEGER;
	journalmaxeventsperentry INTEGER;
	journalservernameserverserial INTEGER;
	journalnode INTEGER;
	journalsummary INTEGER;
	journalalertkey INTEGER;
	journalcustomtext INTEGER;
	maxscopeidjournals INTEGER;
	thisjournalentry CHAR(4080);
	scopeidparentjournalentry CHAR(4080);
	journalcounter INTEGER;

begin

	-- INITIALISE LOCAL VARIABLES
	set propagatettnumber = 1;
	set propagateacknowledged = 1;
	set propagateowneruid = 1;
	set propagateownergid = 1;

	set usescopeidprefix = 1;
	set scopeidprefix = '';
	set usescopeidlabel = 1;
	set usescopeidimpactcause = 0;
	set usescopeidcustomtext = 0;
	set propagatetexttoscopeidparentcause = 0;
	set propagatetexttoscopeidparentimpact = 0;
	set propagatetexttoscopeidparentfirst = 0;
	set propagatetexttoscopeidparentlast = 0;
	set customtext = '';
	set usescopeidsitesaffected = 1;
	set usescopeidnumactivealarms = 0;

	set highcauseweight = 0;
	set highcausetext = 'UNKNOWN';
	set highimpactweight = 0;
	set highimpacttext  = 'UNKNOWN';
	set highseverity = 0;
	set lowfirstoccurrence = 0;
	set highlastoccurrence = 0;
	set alarmcounter= 0;
	set alarmcounter2 = 0;
	set alarmcounter3 = 0;
	set summary = '';

	set journaluid = 0;
	set journaltoscopeidparent = 0;
	set journalmaxeventsperentry = 20;
	set journalservernameserverserial = 1;
	set journalnode = 1;
	set journalsummary = 1;
	set journalalertkey = 0;
	set journalcustomtext = 0;
	set maxscopeidjournals = 50;
	set thisjournalentry = '';
	set scopeidparentjournalentry = '';
	set journalcounter = 0;

	-- LOAD UP VARIABLES BASED ON PROPERTIES
	for each row property in master.properties where property.Name in (
		'SEGPropagateTTNumber',
		'SEGPropagateAcknowledged',
		'SEGPropagateOwnerUID',
		'SEGPropagateOwnerGID',
		'SEGUseScopeIDPrefix',
		'SEGScopeIDPrefix',
		'SEGUseScopeIDLabel',
		'SEGUseScopeIDImpactCause',
		'SEGUseScopeIDCustomText',
		'SEGPropagateTextToScopeIDParentCause',
		'SEGPropagateTextToScopeIDParentImpact',
		'SEGPropagateTextToScopeIDParentFirst',
		'SEGPropagateTextToScopeIDParentLast',
		'SEGUseScopeIDSitesAffected',
		'SEGUseScopeIDNumActiveAlarms',
		'SEGJournalUID',
		'SEGJournalToScopeIDParent',
		'SEGJournalMaxEventsPerEntry',
		'SEGJournalServerNameServerSerial',
		'SEGJournalNode',
		'SEGJournalSummary',
		'SEGJournalAlertKey',
		'SEGJournalCustomText',
		'SEGMaxScopeIDJournals')
	begin

		-- SPECIFIES WHETHER OR NOT TO PROPAGATE THE TTNumber TO CHILD EVENTS
		if (property.Name = 'SEGPropagateTTNumber') then

			set propagatettnumber = property.IntValue;

		-- SPECIFIES WHETHER OR NOT TO PROPAGATE THE Acknowledged STATUS TO CHILD EVENTS
		elseif (property.Name = 'SEGPropagateAcknowledged') then

			set propagateacknowledged = property.IntValue;

		-- SPECIFIES WHETHER OR NOT TO PROPAGATE THE OwnerUID TO CHILD EVENTS
		elseif (property.Name = 'SEGPropagateOwnerUID') then

			set propagateowneruid = property.IntValue;

		-- SPECIFIES WHETHER OR NOT TO PROPAGATE THE OwnerGID TO CHILD EVENTS
		elseif (property.Name = 'SEGPropagateOwnerGID') then

			set propagateownergid = property.IntValue;

		-- SPECIFIES WHETHER OR NOT TO USE THE ScopeID PREFIX IN THE ScopeIDParent EVENT
		elseif (property.Name = 'SEGUseScopeIDPrefix') then

			set usescopeidprefix = property.IntValue;

		-- SPECIFIES THE ScopeID PREFIX TO USE IN THE ScopeIDParent EVENT
		elseif (property.Name = 'SEGScopeIDPrefix') then

			set scopeidprefix = property.CharValue;

		-- SPECIFIES WHETHER OR NOT TO INCLUDE THE ScopeID LABEL IN THE ScopeIDParent EVENT
		elseif (property.Name = 'SEGUseScopeIDLabel') then

			set usescopeidlabel = property.IntValue;

		-- SPECIFIES WHETHER OR NOT TO INCLUDE THE IMPACT AND CAUSE INFO IN THE ScopeIDParent EVENT
		elseif (property.Name = 'SEGUseScopeIDImpactCause') then

			set usescopeidimpactcause = property.IntValue;

		-- SPECIFIES WHETHER OR NOT TO INCLUDE THE CUSTOM TEXT IN THE ScopeIDParent EVENT
		elseif (property.Name = 'SEGUseScopeIDCustomText') then

			set usescopeidcustomtext = property.IntValue;

		-- SPECIFIES WHETHER OR NOT TO PROPAGATE THE CUSTOM TEXT TO THE ScopeIDParent EVENT
		-- BASED ON THE CUSTOM TEXT OF THE CHILD EVENT WITH THE HIGHEST CauseWeight
		elseif (property.Name = 'SEGPropagateTextToScopeIDParentCause') then

			set propagatetexttoscopeidparentcause = property.IntValue;

		-- SPECIFIES WHETHER OR NOT TO PROPAGATE THE CUSTOM TEXT TO THE ScopeIDParent EVENT
		-- BASED ON THE CUSTOM TEXT OF THE CHILD EVENT WITH THE HIGHEST ImpactWeight
		elseif (property.Name = 'SEGPropagateTextToScopeIDParentImpact') then

			set propagatetexttoscopeidparentimpact = property.IntValue;

		-- SPECIFIES WHETHER OR NOT TO PROPAGATE THE CUSTOM TEXT TO THE ScopeIDParent EVENT
		-- BASED ON THE CUSTOM TEXT OF THE CHILD EVENT WITH THE LOWEST FirstOccurrence
		elseif (property.Name = 'SEGPropagateTextToScopeIDParentFirst') then

			set propagatetexttoscopeidparentfirst = property.IntValue;

		-- SPECIFIES WHETHER OR NOT TO PROPAGATE THE CUSTOM TEXT TO THE ScopeIDParent EVENT
		-- BASED ON THE CUSTOM TEXT OF THE CHILD EVENT WITH THE HIGHEST LastOccurrence
		elseif (property.Name = 'SEGPropagateTextToScopeIDParentLast') then

			set propagatetexttoscopeidparentlast = property.IntValue;

		-- SPECIFIES WHETHER OR NOT TO INCLUDE THE NUMBER OF AFFECTED SITES IN THE ScopeIDParent EVENT
		elseif (property.Name = 'SEGUseScopeIDSitesAffected') then

			set usescopeidsitesaffected = property.IntValue;

		-- SPECIFIES WHETHER OR NOT TO INCLUDE THE NUMBER OF ACTIVE ALARMS IN THE ScopeIDParent EVENT
		elseif (property.Name = 'SEGUseScopeIDNumActiveAlarms') then

			set usescopeidnumactivealarms = property.IntValue;

		-- SPECIFIES THE UID TO USE WHEN INSERTING JOURNALS INTO SYNTHETIC PARENT EVENTS - DEFAULT IS ROOT
		elseif (property.Name = 'SEGJournalUID') then

			set journaluid = property.IntValue;

		-- SPECIFIES WHETHER OR NOT TO JOURNAL CHILD EVENTS TO THE ScopeIDParent EVENT
		elseif (property.Name = 'SEGJournalToScopeIDParent') then

			set journaltoscopeidparent = property.IntValue;

		-- SPECIFIES HOW MANY EVENTS TO ROLL UP INTO A SINGLE JOURNAL ENTRY
		elseif (property.Name = 'SEGJournalMaxEventsPerEntry') then

			set journalmaxeventsperentry = property.IntValue;

		-- SPECIFIES WHETHER OR NOT TO SEND THE ServerName:ServerSerial FIELD TO THE PARENT EVENT JOURNAL
		elseif (property.Name = 'SEGJournalServerNameServerSerial') then

			set journalservernameserverserial = property.IntValue;

		-- SPECIFIES WHETHER OR NOT TO SEND THE Node FIELD TO THE PARENT EVENT JOURNAL
		elseif (property.Name = 'SEGJournalNode') then

			set journalnode = property.IntValue;

		-- SPECIFIES WHETHER OR NOT TO SEND THE Summary FIELD TO THE PARENT EVENT JOURNAL
		elseif (property.Name = 'SEGJournalSummary') then

			set journalsummary = property.IntValue;

		-- SPECIFIES WHETHER OR NOT TO SEND THE AlertKey FIELD TO THE PARENT EVENT JOURNAL
		elseif (property.Name = 'SEGJournalAlertKey') then

			set journalalertkey = property.IntValue;

		-- SPECIFIES WHETHER OR NOT TO SEND THE CustomText FIELD TO THE PARENT EVENT JOURNAL
		elseif (property.Name = 'SEGJournalCustomText') then

			set journalcustomtext = property.IntValue;

		-- SPECIFIES MAXIMUM NUMBER OF EVENTS TO SEND TO THE PARENT EVENT JOURNAL
		elseif (property.Name = 'SEGMaxScopeIDJournals') then

			set maxscopeidjournals = property.IntValue;
		end if;
	end;

	-- IN CASE MULTIPLE CustomText PROPAGATION OPTIONS ARE INADVERTENTLY SELECTED,
	-- ENFORCE AN ORDER OF PRECEDENCE TO ENSURE UNDEFINED RESULTS ARE AVOIDED
	--
	-- ORDER OF PRECEDENCE IS:
	-- 1 PROPAGATE CustomText OF HIGHEST CauseWeight
	-- 2 PROPAGATE CustomText OF HIGHEST ImpactWeight
	-- 3 PROPAGATE CustomText OF LOWEST FirstOccurrence
	-- 4 PROPAGATE CustomText OF HIGHEST LastOccurrence
	--
	if (propagatetexttoscopeidparentcause = 1) then

		-- CLEAR OTHER SELECTED OPTIONS
		set propagatetexttoscopeidparentimpact = 0;
		set propagatetexttoscopeidparentfirst = 0;
		set propagatetexttoscopeidparentlast = 0;

	elseif (propagatetexttoscopeidparentimpact = 1) then

		-- CLEAR OTHER SELECTED OPTIONS
		set propagatetexttoscopeidparentfirst = 0;
		set propagatetexttoscopeidparentlast = 0;

	elseif (propagatetexttoscopeidparentfirst = 1) then

		-- CLEAR OTHER SELECTED OPTIONS
		set propagatetexttoscopeidparentlast = 0;

	end if;

	-- UPDATE EACH ScopeIDParent EVENT
	for each row scopeid in alerts.status where scopeid.AlertGroup = 'ScopeIDParent'
	begin

		-- RESET LOCAL VARIABLES
		-- alarmcounter: USED TO COUNT UP NUMBER OF AFFECTED SITES
		-- alarmcounter2: USED TO COUNT UP NUMBER OF SITE EVENT CHILDREN
		-- alarmcounter3: USED TO COUNT UP NUMBER OF EVENTS DIRECTLY UNDER ScopeIDParent
		set highcauseweight = 0;
		set highcausetext = 'UNKNOWN';
		set highimpactweight = 0;
		set highimpacttext  = 'UNKNOWN';
		set highseverity = 0;
		set lowfirstoccurrence = 0;
		set highlastoccurrence = 0;
		set customtext = '';
		set alarmcounter  = 0;
		set alarmcounter2  = 0;
		set alarmcounter3  = 0;
		set summary = '';
		set thisjournalentry = '';
		set scopeidparentjournalentry = '';
		set journalcounter = 0;

		-- EXAMINE EACH CHILD OF THE CURRENT ScopeIDParent EVENT
		for each row child in alerts.status where
			child.ParentIdentifier = scopeid.Identifier
		begin

			-- ONLY CONSIDER NON-CLEARED CHILD EVENTS
			if (child.Severity > 0) then

				-- INCLUDE CURRENT CHILD EVENT IN COUNTS
				-- PROCESS SiteNameParent EVENTS
				if (child.AlertGroup = 'SiteNameParent') then

					-- EXCLUDE SiteNameParent EVENTS THAT HAVE NO SiteName
					if (child.SiteName != 'NO SITENAME') then

						-- INCREMENT NUMBER OF AFFECTED SITES
						set alarmcounter = alarmcounter + 1;
					end if;

					-- ADD TO EVENT COUNTER THE NUMBER OF UNDERLYING EVENTS
					set alarmcounter2 = alarmcounter2 + child.Grade;

				-- ELSE EVENT IS A REAL EVENT AND NOT A SYNTHETIC ONE
				else

					-- STORE THE HIGHEST CauseWeight OF THE CHILDREN EVENTS
					-- AND ITS ASSOCIATED NormalisedAlarmName
					if (child.CauseWeight > highcauseweight) then
	
						set highcauseweight = child.CauseWeight;
						set highcausetext = child.NormalisedAlarmName;

						-- IF OPTION TO PROPAGATE CUSTOM TEXT TO ScopeIDParent EVENT FROM HIGHEST
						-- CauseWeight CHILD IS ENABLED, STORE CURRENT EVENT CUSTOM TEXT
						if (propagatetexttoscopeidparentcause = 1) then

							set customtext = child.CustomText;
						end if;
					end if;

					-- STORE THE HIGHEST ImpactWeight OF THE CHILDREN EVENTS
					-- AND ITS ASSOCIATED NormalisedAlarmName
					if (child.ImpactWeight > highimpactweight) then
	
						set highimpactweight = child.ImpactWeight;
						set highimpacttext = child.NormalisedAlarmName;

						-- IF OPTION TO PROPAGATE CUSTOM TEXT TO ScopeIDParent EVENT FROM HIGHEST
						-- ImpactWeight CHILD IS ENABLED, STORE CURRENT EVENT CUSTOM TEXT
						if (propagatetexttoscopeidparentimpact = 1) then

							set customtext = child.CustomText;
						end if;
					end if;

					-- INCREMENT EVENT COUNTER
					set alarmcounter3 = alarmcounter3 + 1;
				end if;

				-- STORE THE HIGHEST Severity OF THE CHILDREN EVENTS
				if (highseverity < child.Severity) then

					set highseverity = child.Severity;
				end if;

				-- STORE THE LOWEST NON-ZERO FirstOccurrence OF THE CHILDREN EVENTS
				if (lowfirstoccurrence > child.FirstOccurrence or lowfirstoccurrence = 0) then

					set lowfirstoccurrence = child.FirstOccurrence;

					-- IF OPTION TO PROPAGATE CUSTOM TEXT TO ScopeIDParent EVENT FROM LOWEST
					-- FirstOccurrence CHILD IS ENABLED, STORE CURRENT EVENT CUSTOM TEXT
					if (propagatetexttoscopeidparentfirst = 1) then

						set customtext = child.CustomText;
					end if;
				end if;

				-- STORE THE HIGHEST LastOccurrence OF THE CHILDREN EVENTS
				if (highlastoccurrence < child.LastOccurrence) then

					set highlastoccurrence = child.LastOccurrence;

					-- IF OPTION TO PROPAGATE CUSTOM TEXT TO ScopeIDParent EVENT FROM HIGHEST
					-- LastOccurrence CHILD IS ENABLED, STORE CURRENT EVENT CUSTOM TEXT
					if (propagatetexttoscopeidparentlast = 1) then

						set customtext = child.CustomText;
					end if;
				end if;

				-- PROPAGATE THE TICKET NUMBER FROM THE ScopeIDParent EVENT TO
				-- THE CURRENT CHILD IF THE CURRENT CHILD IS UNTICKETED,
				-- UNSUPPRESSED AND THE OPTION TO PROPAGATE TICKET NUMBER IS ENABLED
				if (scopeid.TTNumber != '' and child.TTNumber = '' and
					child.SuppressEscl != 4 and propagatettnumber = 1) then

					set child.TTNumber = scopeid.TTNumber;
				end if;

				-- PROPAGATE THE ACKNOWLEDGED STATUS FROM THE ScopeIDParent EVENT TO
				-- THE CURRENT CHILD IF THE CURRENT CHILD IS UNACKNOWLEDGED AND THE
				-- OPTION TO PROPAGATE ACKNOWLEDGED STATUS IS ENABLED
				if (scopeid.Acknowledged = 1 and child.Acknowledged != 1
					and propagateacknowledged = 1) then

					set child.Acknowledged = scopeid.Acknowledged;
				end if;

				-- PROPAGATE THE OwnerUID FROM THE ScopeIDParent EVENT TO THE
				-- CURRENT CHILD EVENT IF THE OWNER OF THE ScopeIDParent EVENT
				-- IS DIFFERENT TO THAT OF THE CHILD EVENT AND THE
				-- OPTION TO PROPAGATE OwnerUID IS ENABLED
				if (scopeid.OwnerUID != child.OwnerUID and propagateowneruid = 1) then

					set child.OwnerUID = scopeid.OwnerUID;
				end if;

				-- PROPAGATE THE OwnerGID FROM THE ScopeIDParent EVENT TO THE
				-- CURRENT CHILD EVENT IF THE GROUP OF THE ScopeIDParent EVENT
				-- IS DIFFERENT TO THAT OF THE CHILD EVENT AND THE
				-- OPTION TO PROPAGATE OwnerGID IS ENABLED
				if (scopeid.OwnerGID != child.OwnerGID and propagateownergid = 1) then

					set child.OwnerGID = scopeid.OwnerGID;
				end if;

				-- CLEAR thisjournalentry VARIABLE
				set thisjournalentry = '';

				-- CONSTRUCT A JOURNAL ENTRY FOR CURRENT EVENT IF
				-- - THE CURRENT EVENT HAS NOT ALREADY BEEN JOURNALED
				-- - IF SENDING CHILD EVENT DATA AS JOURNALS TO ScopeIDParent IS ENABLED
				-- - IF THE MAXIMUM NUMBER OF EVENTS PER JOURNAL ENTRY HAS NOT BEEN EXCEEDED
				-- - IF THE MAXIMUM NUMBER OF EVENTS FOR THIS ScopeIDParent HAS NOT BEEN EXCEEDED
				-- - THE CURRENT EVENT IS NOT A SiteNameParent EVENT
				if (child.JournalSent = 0 and journaltoscopeidparent = 1 and
					journalcounter < journalmaxeventsperentry and
					(scopeid.Poll + journalcounter) < maxscopeidjournals and
					child.AlertGroup != 'SiteNameParent') then

					-- CHECK WHETHER OR NOT TO INCLUDE THE ServerName:ServerSerial IN THIS JOURNAL ENTRY
					if (journalservernameserverserial = 1) then

						set thisjournalentry = thisjournalentry + child.ServerName +
							':' + to_char(child.ServerSerial);
					end if;

					-- CHECK WHETHER OR NOT TO INCLUDE THE Node IN THIS JOURNAL ENTRY
					if (journalnode = 1 and child.Node != '') then

						-- ADD COLON BEFORE APPENDING TEXT TO THIS JOURNAL ENTRY
						if (thisjournalentry != '') then
							set thisjournalentry = thisjournalentry + ': ';
						end if;

						set thisjournalentry = thisjournalentry + child.Node;
					end if;

					-- CHECK WHETHER OR NOT TO INCLUDE THE Summary IN THIS JOURNAL ENTRY
					if (journalsummary = 1 and child.Summary != '') then

						-- ADD COLON BEFORE APPENDING TEXT TO THIS JOURNAL ENTRY
						if (thisjournalentry != '') then
							set thisjournalentry = thisjournalentry + ': ';
						end if;

						set thisjournalentry = thisjournalentry + child.Summary;
					end if;

					-- CHECK WHETHER OR NOT TO INCLUDE THE AlertKey IN THIS JOURNAL ENTRY
					if (journalalertkey = 1 and child.AlertKey != '') then

						-- ADD COLON BEFORE APPENDING TEXT TO THIS JOURNAL ENTRY
						if (thisjournalentry != '') then
							set thisjournalentry = thisjournalentry + ': ';
						end if;

						set thisjournalentry = thisjournalentry + child.AlertKey;
					end if;

					-- CHECK WHETHER OR NOT TO INCLUDE THE CustomText IN THIS JOURNAL ENTRY
					if (journalcustomtext = 1 and child.CustomText != '') then

						-- ADD COLON BEFORE APPENDING TEXT TO THIS JOURNAL ENTRY
						if (thisjournalentry != '') then
							set thisjournalentry = thisjournalentry + ': ';
						end if;

						set thisjournalentry = thisjournalentry + child.CustomText;
					end if;
				end if;

				-- ADD CURRENT EVENT TO ScopeIDParent JOURNAL ENTRY IF:
				-- - IF SENDING CHILD EVENT DATA AS JOURNALS TO ScopeIDParent IS ENABLED
				-- - IF THE MAXIMUM NUMBER OF EVENTS FOR THIS ScopeIDParent HAS NOT BEEN EXCEEDED
				-- - THE CURRENT JOURNAL ENTRY IS NOT NULL
				-- - THE CURRENT EVENT IS NOT A SiteNameParent EVENT
				if (journaltoscopeidparent = 1 and
					(scopeid.Poll + journalcounter) < maxscopeidjournals and
					thisjournalentry != '' and child.AlertGroup != 'SiteNameParent') then

					-- APPEND A NEW LINE CHARACTER IF THERE IS ALREADY TEXT TO ADD
					if (scopeidparentjournalentry != '') then
						set scopeidparentjournalentry = scopeidparentjournalentry + '\n';
					end if;

					-- ADD THE CURRENT JOURNAL ENTRY TO THE EXISTING
					set scopeidparentjournalentry = scopeidparentjournalentry + thisjournalentry;

					-- INCREMENT THE NUMBER OF JOURNALS BEING SENT
					set journalcounter = journalcounter + 1;

					-- MARK CHILD EVENT AS HAVING BEEN SENT
					set child.JournalSent = 1;
 				end if;
			end if;
		end;

		-- STORE THE NUMBER OF ACTIVE CHILD EVENTS IN THE GRADE FIELD, IF NOT ZERO
		if (alarmcounter2 != 0 or alarmcounter3 != 0) then

			set scopeid.Grade = alarmcounter2 + alarmcounter3;

		-- ELSE THE COUNT OF ACTIVE CHILD EVENTS IS ZERO, SO RESET GRADE FIELD TO ZERO
		elseif (scopeid.Grade != 0) then

			set scopeid.Grade = 0;
		end if;

		-- BEGIN THE CONSTRUCTION OF THE ScopeIDParent Summary FIELD

		-- SPECIFY WHETHER TO INCLUDE THE ScopeIDParent PREFIX
		if (usescopeidprefix = 1) then

			set summary = scopeidprefix;
		end if;

		-- SPECIFY WHETHER TO INCLUDE THE ScopeID LABEL
		if (usescopeidlabel = 1) then

			-- APPEND A COLON IF summary NOT BLANK
			if (summary != '') then
				set summary = summary + ': ';
			end if;

			set summary = summary + scopeid.ScopeID;
		end if;

		-- SPECIFY WHETHER TO INCLUDE THE CAUSE AND IMPACT TEXT
		-- DO NOT USE IF THERE ARE NO DIRECT CHILD EVENTS
		if (usescopeidimpactcause = 1 and alarmcounter3 != 0) then

			-- APPEND A COLON IF summary NOT BLANK
			if (summary != '') then
				set summary = summary + ': ';
			end if;

			-- APPEND CAUSE AND IMPACT TEXT
			if (highcausetext = 'UNKNOWN' and highimpacttext = 'UNKNOWN') then
				set summary = summary + 'CAUSE AND IMPACT: UNKNOWN';
			elseif (highcausetext = 'UNKNOWN' and highimpacttext != 'UNKNOWN') then
				set summary = summary + highimpacttext + ' UNKNOWN CAUSE';
			elseif (highcausetext != 'UNKNOWN' and highimpacttext = 'UNKNOWN') then
				set summary = summary + highcausetext + ' UNKNOWN IMPACT';
			elseif (highcausetext != 'UNKNOWN' and highimpacttext = highcausetext) then
				set summary = summary + 'CAUSE AND IMPACT: ' + highcausetext;
			else
				set summary = summary + highimpacttext +
					' caused by ' + highcausetext;
			end if;
		end if;

		-- SPECIFY WHETHER TO INCLUDE THE CUSTOM SUMMARY TEXT CustomText
		if (usescopeidcustomtext = 1 and scopeid.CustomText != '') then

			-- APPEND A COLON IF summary NOT BLANK
			if (summary != '') then
				set summary = summary + ': ';
			end if;

			-- APPEND CustomText
			set summary = summary + scopeid.CustomText;
		end if;

		-- SPECIFY WHETHER TO INCLUDE THE NUMBER OF SITES AFFECTED
		-- DO NOT INCLUDE IF NO SITES ARE AFFECTED
		if (usescopeidsitesaffected = 1 and alarmcounter != 0) then

			-- APPEND A COLON IF summary NOT BLANK
			if (summary != '') then
				set summary = summary + ': ';
			end if;

			-- APPEND NUMBER OF SITES AFFECTED
			set summary = summary + to_char(alarmcounter) + ' site';

			if (alarmcounter = 1) then
				set summary = summary + ' affected';
			else
				set summary = summary + 's affected';
			end if;
		end if;

		-- SPECIFY WHETHER TO INCLUDE THE NUMBER OF ACTIVE ALARMS IN THE GROUPING
		if (usescopeidnumactivealarms = 1) then

			-- APPEND A SPACE CHARACTER IF summary NOT BLANK
			if (summary != '') then
				set summary = summary + ' ';
			end if;

			-- APPEND TOTAL NUMBER OF UNDERLYING REAL EVENTS
			if ((alarmcounter2 + alarmcounter3) = 1) then
				set summary = summary + '(' +
					to_char(alarmcounter2 + alarmcounter3) + ' active alarm)';
			else
				set summary = summary + '(' +
					to_char(alarmcounter2 + alarmcounter3) + ' active alarms)';
			end if;
		end if;

		-- UPDATE ScopeIDParent EVENT IF NOT CLEAR - OR IF ANY OF THE COUNTERS ARE NOT ZERO
		if (scopeid.Severity != 0 or alarmcounter != 0 or alarmcounter2 != 0 or alarmcounter3 != 0) then

			-- UPDATE Summary FIELD
			set scopeid.Summary = summary;

			-- SET THE Severity TO INDETERMINITE IF UNDERLYING EVENTS ARE CLEARED
			-- THIS WILL EVENTUALLY BE CLEARED WHEN TIME WINDOW CLOSES
			if (highseverity = 0) then

				set scopeid.Severity = 1;

			-- ELSE SET THE Severity TO THAT OF THE HIGHEST UNDERLYING EVENT
			else

				set scopeid.Severity = highseverity;
			end if;

			-- UPDATE FirstOccurrence BASED ON UNDERLYING EVENTS, IF NOT NULL
			if (lowfirstoccurrence != 0) then

				set scopeid.FirstOccurrence = lowfirstoccurrence;
			end if;

			-- UPDATE LastOccurrence BASED ON UNDERLYING EVENTS, IF NOT NULL
			if (highlastoccurrence != 0) then

				set scopeid.LastOccurrence = highlastoccurrence;
			end if;

			-- UPDATE CustomText FIELD OF THE PARENT IF ONE OF THE PROPAGATION
			-- FROM CHILD OPTIONS ARE ENABLED
			if (	propagatetexttoscopeidparentcause = 1 or
				propagatetexttoscopeidparentimpact = 1 or
				propagatetexttoscopeidparentfirst = 1 or
				propagatetexttoscopeidparentlast = 1) then

				set scopeid.CustomText = customtext;
			end if;

		end if;

		-- ADD A JOURNAL ENTRY TO THE CURRENT ScopeIDParent IF APPLICABLE
		if (journaltoscopeidparent = 1 and scopeidparentjournalentry != '') then

			-- ADD THE ROLLED UP CHILDREN EVENTS TO THE CURRENT SiteNameParent AS A JOURNAL ENTRY
			EXECUTE jinsert(scopeid.Serial, journaluid, getdate(),
				'SCOPEID CHILD EVENTS:\n' + scopeidparentjournalentry);

			-- UPDATE THE NUMBER OF CHILD EVENTS APPENDED TO THIS PARENT EVENT
			set scopeid.Poll = scopeid.Poll + journalcounter;

			-- APPEND NOTE THAT NO FURTHER JOURNALS WILL BE APPENDED TO THIS ScopeIDParent
			-- DUE TO THE MAXIMUM NUMBER HAVING BEEN REACHED
			if (scopeid.Poll = maxscopeidjournals) then

				EXECUTE jinsert(scopeid.Serial, journaluid, getdate() + 1,
					'*** MAXIMUM OF ' + to_char(maxscopeidjournals) +
					' EVENTS HAVE BEEN JOURNALED TO THIS PARENT');
			end if;
		end if;
	end;
end;
go

------------------------------------------------------------------------------
-- CREATE A TRIGGER TO UPDATE EXISTING SYNTHETIC PARENT EVENTS
------------------------------------------------------------------------------

CREATE OR REPLACE TRIGGER correlation_process_existing_parents
GROUP correlation_triggers
PRIORITY 15
COMMENT 'Update any existing synthetic parent events that are present'
EVERY 17 SECONDS
WHEN get_prop_value('ActingPrimary') %= 'TRUE'
begin

-- STEP 1: DISMANTLE ANY SMALL GROUPS THAT HAVE EXPIRED AND DON'T HAVE MINIMUM NUMBER OF CHILDREN

	EXECUTE correlation_dismantle_small_groups;

-- STEP 2: CLEAR PARENT EVENTS THAT HAVE NO CHILDREN AND ARE PAST THEIR EXPIRY TIMES

	EXECUTE correlation_clear_expired_parents;

-- STEP 3: PROCESS SiteNameParent EVENTS

	EXECUTE correlation_process_sitenameparents;

-- STEP 4: PROCESS ScopeIDParent EVENTS

	EXECUTE correlation_process_scopeidparents;

end;
go

------------------------------------------------------------------------------
-- REMOVE DEFUNCT TABLE master.correlation_scopealias

DELETE FROM master.correlation_scopealias;
go

DROP TABLE master.correlation_scopealias;
go

